
// This script contains a demonstration code at bottom. So it can be also
// simply executed to see the function in action on various examples.

// IMPORTANT: The function HexCopy as well as the code for demonstrating
//            the function is made compatible as much as possible also for
//            UltraEdit for Windows v13.00 and UES v6.20. But because of some
//            serious problems of the scripting engine in these versions, is
//            strongly recommended to use at least UE for Windows v13.20a+1
//            and UES v6.40a+1 for this script as well as for any other script.

// === HexCopy ===============================================================

/* Script Name:   HexCopy.js
   Creation Date: 2010-01-08
   Last Modified: 2022-05-29
   Copyright:     Copyright (c) 2022 by Mofi
   Original:      https://www.ultraedit.com/resources/scripts/HexCopy.zip
                  http://forums.ultraedit.com/viewtopic.php?f=52&t=6039

The function HexCopy makes by default the same as command Hex Copy Selected
View in menu Edit - Hex Functions. It copies the current selection in hex
edit mode to the active clipboard. But in comparison to the original command
it selects by default everything if nothing is currently selected. And it
works also for selections in normal text edit mode. Furthermore it offers
some additional options as listed below to determine the output format
and various input sources and output destinations can be specified too.

The function selects automatically by default the entire file content
if nothing is currently selected. It does not matter if the active file
is opened in text edit mode or in hex edit mode. So it works for text
files as well as for binary files.

Depending on the options the function can be used for converting a binary
file to ASCII or to get the hexadecimal values of a selected text in ASCII.

Also possible is to get the hex values of a string passed to the function.

The function can be used also to convert the clipboard content to an ASCII
string with hexadecimal values. Converting the clipboard content works only
with UltraEdit for Windows >= v14.20 and UEStudio >= v9.00.

The function uses automatically 16 bit hexadecimal values if any character
is a Unicode encoded character (character code is greater than decimal 255).
Otherwise the output is always with 8 bit hexadecimal values. Selections
in hex edit mode produce always 8 bit hexadecimal values even if the file
is a Unicode encoded text file. All characters with a value greater 255
are printed in the ASCII/ANSI output with a point.

The function returns   false   if successful operation was not possible
like source string or clipboard content is empty or nothing is selected
and the entire file content should not be selected in this case. Enable
the debug mode by defining a global variable named g_nDebugMessage with
value 1 (print messages to output window) or 2 (message pop-ups) if the
reason should be printed why the function returns false.

Note: Using first parameter with number  -2  results in returning "" (empty
      string) instead of the boolean value  false  on an error condition.

All parameters are optional. So it is possible to call the function also
without any parameter. Calling the function without any parameters results
in a replacement for command Hex Copy Selected View with the same content
in the clipboard because the initial demand was to write a function for
this command not available for scripts and macros.

A default value is used for every parameter not specified on function
call. Therefore only those parameters must be specified that are left
the last parameter which should not have a default value.

Most of the parameters support a boolean value, a number or a string,
see the detailed description about the optional parameters below.

As input parameters the function offers following options:

  1) Output - the default values  -4  (number) or  false  (boolean)
              result in copying the output to the active clipboard.

     The boolean value  true  for this option writes the output to a
     new file instead of the active clipboard resulting in not modifying
     current clipboard content. The number  -1  can be also specified to
     write to a new file. The new file is active on function exit and the
     cursor is at bottom of the new file.

     The number  -2  tells the function to return the string to the calling
     function. In this case an empty string is returned instead of boolean
     value  false  on an error.

     The number  -3  appends the output to the current content of the active
     clipboard instead of replacing the clipboard content with the output.

     Any number lower  -3  results in default behavior of copying the
     output to the active clipboard.

     Number  0  or any positive number is interpreted by the function as
     request to write the output to the file with document index equal the
     number. Positive numbers greater or equal the number of open documents
     results in writing to a new file with a warning if debug messages are
     enabled. Please note that file 1 has document index number 0.

     A string as first parameter is interpreted also as request to write
     to a new file and causes execution of the Save As command using the
     string value as file name. The Save As dialog is opened during the
     execution if the string is empty or is just a question mark. No
     check is done if the file name is valid and saving the new file
     really worked.


  2) DataSource - the default values  0  (number) or  false  (boolean)
                  result in working on the current selection with an
                  automatic selection of the entire file content if
                  nothing is currently selected.

     The command Hex Copy Selected View is disabled when hex edit mode is
     not active and also when nothing is selected. But the author of this
     function found it annoying to use always the command Select All before
     the command Hex Copy Selected View could be used which is the reason
     why there is this difference in default behavior of function HexCopy.

     Following numbers are possible:

        -1 ... use clipboard content (-1 or any negative number).

         0 ... use current selection directly with selecting
               everything if nothing is currently selected (default).

         1 ... use current selection directly, but do not select
               everything if nothing is currently selected.

         2 ... like option 0, but copy the selection first to the active
               clipboard and then store the clipboard content into the
               string variable instead of storing the selection directly
               into the string variable. In comparison to behavior of
               option 0 this produces the correct result also for files
               opened in normal text edit mode containing Unicode encoded
               characters which are not listed in currently used code page.

         3 ... like option 1, but with using temporarily the clipboard as
               explained above for option 2 too.

     Using the function with the options 2, 3 or any number lower 0 requires
     UltraEdit for Windows >= v14.20 and UES >= v9.00. False or an empty
     string is returned when the used application does not support the
     required object for these options.

     It is also possible to pass a boolean value. Value  false  is equal
     option 0 and value  true  is interpreted as option -1.

     If a string is passed as parameter, the function uses this
     string as input data source.


  3) Terminate - with the default values  0  (number) or  false  (boolean)
                 no line termination is appended on last line of the output.

     A line termination is appended with a number not equal 0 or with
     boolean value  true.

     The function also automatically appends spaces at end of the last
     line when appending a line termination to force all lines having
     the same length. Last line can be shorter than the other lines if
     default behavior of last line without line termination is used.

     It is also possible to pass an empty string instead of  0  or  false.
     The passed string itself is appended if the string is not empty. This
     can be used to automatically append a footer. The function does not
     insert or append the line termination if this string does not include
     one.


  4) HexValues - the default value 16 results in the appropriate number
                 of hexadecimal values per line.

     The maximum number of values per line is 512, the minimum is 1.
     The function uses the default value if the number is out of range
     and prints a warning message if debug messages are enabled.

     This parameter must be a number. Strings or booleans are always
     ignored for this parameter without a warning.

     It is not possible from within a script to access configuration
     settings like "Number of HEX characters per line". So if this
     configuration setting has not the default value 16 and therefore
     this function should also work with a different default value,
     this number must be changed at top of the function, see below.


  5) HexFormat - the default values  1  (number) or  true  (boolean)
                 result in inserting a single space character left
                 of every hexadecimal value.

     A number not equal 0 is also interpreted as true. Number  0  or
     boolean value  false  for this parameter disables writing a space
     left every hexadecimal value.

     Also possible is to pass an empty string for this option which
     is interpreted as boolean value  false  or a string with a
     single character which is used left every hexadecimal value.

     A string containing %X is interpreted as output format string.
     Everything left %X is printed left to a hexadecimal value and
     everything right %X is printed right to a hexadecimal value. The
     last character right %X is not printed for the last hex value. So it
     is possible for example to use "0x%X," to get the hex values as needed
     for many programming languages like C/C++, C#, JScript, ... or to use
     "0%Xh," for assembler. For these examples the last hexadecimal value
     has no comma appended. Also possible is to use lowercase %x which
     results in lowercase letters a-f instead of default A-F. Therefore
     using " %x" or "%x" can be used to get the default behavior with
     lowercase hexadecimal letters.

     Other strings with a length greater 1 are interpreted also like
     boolean value  false  without any warning. So for example "no space"
     can be used in the function call which is helpful for better reading
     the parameters.


  6) OffsetInfo - the default values  1  (number) or  true  (boolean) result
                  in inserting the file/string offset information at start
                  of every line using the same format as in hex edit mode
                  which is "%08xh: ".

     The number  0  or boolean value  false  for this parameter disables
     writing the offset information which is normally not important when
     using this function for a selected string in a file opened in normal
     text edit mode. Any number not equal 0 is interpreted as  true  too.

     File offset for a selection in a file not starting at file offset 0 is
     only correct with UltraEdit for Windows >= v13.10 and UEStudio >= v6.30.
     A warning is printed if debug messages are enabled and the application
     does not support the required property to get the file position when
     not the entire file is selected by the function itself.

     An empty string is interpreted as  false  and any other string as
     true. If the string contains %X or %x everything left this format
     specifier is written left the hexadecimal offset with automatic correct
     number of required leading zeros for identical lengths of all offsets
     and without any additional characters and everything right is appended
     right to the offset string. %X is for hexadecimal offset in uppercase */
//   and %x for lowercase. For example using "/* %xh: */ " can be used to
/*   comment the offset information when using this function to create a
     C/C++ array with hexadecimal values.

     Other not empty strings like "with offset" in function call are
     possibly better than just  1  or  true  when reading the source
     code of a script using this function.

     Note: The author has not tested what happens when offset is greater
           2.147.483.647 (= FFFFFFFFh = maximum signed 32-bit integer).


  7) AsText - the default values  1  (number) or  true  (boolean) produce
              ASCII/ANSI representation of the bytes in the output like
              displayed in hex edit mode.

     For normal text this output is often not necessary and therefore it
     can be disabled with number  0  or boolean value  true. Any number
     not equal 0 is also interpreted as  true  for this parameter.

     An empty string is interpreted as  false  and any other string as
     true. Similar to behavior of parameter OffsetInfo with %X, a string
     containing %s (only lowercase) is interpreted as format specifier. */
//   For example " /* %s */" can be used to add as comment the ASCII/ANSI
/*   representation of the bytes when using this function to build a
     JavaScript array with hexadecimal values.

     Other not empty strings like "text display" in function call are
     possibly better than just  1  or  true  when reading the source
     code of a script using this function.


  8) LineTerm - the default values are  0  or  false  or  "\r\n".

     With a number the type of the line termination can be determined.
     Four values are possible:

        0 ... "\r\n" ... DOS  ... carriage return and line-feed (default)
        1 ... "\n"   ... UNIX ... only line-feed
        2 ... "\r"   ... MAC  ... only carriage return
        3 ... "?"    ... same as in source (auto-detect). The line
              termination is DOS if no line ending characters are
              found in the source data.

     Don't use value 3 for binary files opened in hex edit mode. The
     line termination type may be unpredictable for binary files and
     therefore the function uses DOS if it can determine if the source
     is from a selection in a file opened currently in hex edit mode.

     It is also possible to specify this parameter as string with the
     values displayed above. Other strings or numbers are ignored with
     a warning if the debug messages are enabled.

     A boolean value can be used also with value  false  for DOS and
     value  true  for UNIX.

     When writing the output to an already opened file specify the line
     termination type required for this file. The function does not check
     if the line type of the output is identical with the line type of the
     destination file. For new files the line type is set by the function.

Note: For maximum speed everything is done in memory. Therefore don't use
      this function on huge selections with hundreds or thousands of MBs.
      But it is possible to select parts of a large or huge file and call
      this function with writing the output to a file in a loop to bypass
      a possible memory problem.

This function is copyrighted by Mofi for free usage by UE/UES users. The
author cannot be made responsible for any damage caused by this function.
You use it at your own risk. */

function HexCopy (Output, DataSource, Terminate, HexValues, HexFormat, OffsetInfo, AsText, LineTerm)
{
   // Variables which can be modified by the input parameters.
   var nHexCount = 16;        // number of hexadecimal values per line
   var nDestination = -4;     // destination for the output (clipboard)
   var sLastTerm = "";        // line termination on last line (none)
   var sFileName = "";        // name of the file when saving new file
   var sHexValLeft = " ";     // string left the hexadecimal values
   var sHexValRight = "";     // string right the hexadecimal values
   var sOffsetLeft = "";      // string left file/string offset information
   var sOffsetRight = "h: ";  // string right file/string offset information
   var sTextLeft = " ; ";     // string left text representation
   var sTextRight = "";       // string right text representation
   var sLineEnd = "\r\n";     // line ending character(s)

   // Internal variables used by the function
   var nIndex = 0;            // current index in arrays
   var nInput = 0;            // holds the number for the input source
   var nOffset = 0;           // current offset in the file/string
   var nCharCode = 0;         // value of the current character (byte)
   var nHexDigits = 8;        // number of digits for offset information
   var nValDigits = 2;        // number of digits per hexadecimal number
   var nValueCount = 0;       // current number of values in the line
   var nDigitsCount = 0;      // number of digits of the current value string
   var nWriteOffset = 1;      // write offset information to output
   var sOutputText = "";      // string for the binary data as ASCII
   var sHexValue = "";        // hexadecimal string of a character value
   var sSourceData = "";      // holds the source data stream (string)
   var sCharacters = "";      // holds the bytes (characters) as characters
   var sOutputLine = "";      // holds the current output line
   var bMacWrite = false;     // write to file with MAC line endings
   var bWriteText = true;     // write as characters to output
   var bOffsetUpper = false;  // offset information in uppercase
   var bValuesUpper = true;   // hexadecimal values in uppercase
   var bSpecialChar = false;  // byte with value 0x80 to 0x9F in data stream

   /* Determine the type of output for debug messages from the global
      variable g_nDebugMessage: 1 ... debug to output window, 2 ... debug
      to message dialog, all others ... no debug messages. If the global
      variable g_nDebugMessage does not exist, display no debug messages. */
   var nOutputType = (typeof(g_nDebugMessage) == "number") ? g_nDebugMessage : 0;

   // Debug messages are not possible with UE for Windows v13.00 and UES v6.20.
   if (!UltraEdit.messageBox) nOutputType = 0;

   // ===== Output =====

   // Evaluate the optional parameters according to the description above
   // starting with the output destination because it defines the type
   // of the return value.
   if (typeof(Output) == "number")    // Is the output parameter a number?
   {
      // Return output as string to the calling routine?
      if (Output == -2)
      {
         // Return type is string with an empty string for an error.
         var ErrorCode = "";
      }
      else   // All other numbers.
      {
         // Return type is boolean with error value false.
         var ErrorCode = false;

         // Write to an already opened file?
         if (Output >= 0)
         {
            // Is the document index valid?
            if (Output >= UltraEdit.document.length)
            {
               if (nOutputType == 2)
               {
                  UltraEdit.messageBox("Passed document index "+Output+" invalid, writing to new file!","HexCopy Warning");
               }
               else if (nOutputType == 1)
               {
                  if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
                  UltraEdit.outputWindow.write("HexCopy: Passed document index "+Output+" invalid, writing to new file!");
               }
               Output = -1;  // Invalid! Write to new file.
            }
         }
      }
      nDestination = Output;
   }
   else  // Output parameter is string, boolean or undefined.
   {
      // Return type is boolean with error value false.
      var ErrorCode = false;

      // Is the output parameter a string?
      if (typeof(Output) == "string")
      {
         // Write output to a new file and save it with the passed file
         // name or display the Save As dialog if string is empty or
         // contains just a question mark.
         nDestination = -1;
         sFileName = (Output.length) ? Output : "?";

      // Is the output parameter a boolean?
      }
      else if (typeof(Output) == "boolean")
      {
         // True ... write to new file, false ... copy to clipboard.
         nDestination = Output ? -1 : -4;
      }
   }

   // ===== DataSource =====

   // Next the most important parameter is evaluated - the data source. This
   // is the the only parameter which can cause the function to exit before
   // really starting. All returns with error code are in the following code
   // block.
   if (typeof(DataSource) == "string") // Is the data source a string?
   {
      if (!DataSource.length)          // Is the string empty?
      {
         if (nOutputType == 2)
         {
            UltraEdit.messageBox("Passed source string is empty!","HexCopy Error");
         }
         else if (nOutputType == 1)
         {
            if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
            UltraEdit.outputWindow.write("HexCopy: Passed source string is empty!");
         }
         return ErrorCode;             // Nothing to do for this function.
      }
      sSourceData = DataSource;        // Passed string is not empty.
      nInput = 4;   // No further actions required to get source string.
   }
   else if (typeof(DataSource) == "number")
   {
      // Values lower 0 and the values 0, 1, 2 and 3 are valid.
      if (DataSource < 4) nInput = DataSource;
      else
      {
         if (nOutputType == 2)         // Ignore invalid value for data source parameter.
         {
            UltraEdit.messageBox("Invalid value "+DataSource+" for parameter DataSource ignored!","HexCopy Warning");
         }
         else if (nOutputType == 1)
         {
            if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
            UltraEdit.outputWindow.write("Invalid value "+DataSource+" for parameter DataSource ignored!");
         }
      }
   }
   else if (typeof(DataSource) == "boolean")
   {
      // Boolean value true selects the clipboard as source, and value
      // false means use current selection with automatic selecting and
      // direct string copy from the selection into a string variable.
      if (DataSource) nInput = -1;
   }

   if (nInput < 0)    // Get string from active clipboard.
   {
      // Is this object supported by this version of the application?
      if (typeof(UltraEdit.clipboardContent) == "undefined")
      {
         if (nOutputType == 2)
         {
            UltraEdit.messageBox("Value "+DataSource+" for parameter DataSource not possible because UltraEdit.clipboardContent not supported!","HexCopy Error");
         }
         else if (nOutputType == 1)
         {
            if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
            UltraEdit.outputWindow.write("HexCopy: Value "+DataSource+" for parameter DataSource not possible because UltraEdit.clipboardContent not supported!");
         }
         return ErrorCode;             // Clipboard as source is not possible.
      }

      // No problem to get content from the clipboard.
      sSourceData = UltraEdit.clipboardContent;

      if (!sSourceData.length)         // Is the active clipboard empty?
      {
         if (nOutputType == 2)
         {
            if (!UltraEdit.clipboardIdx)
            {
               UltraEdit.messageBox("The Windows clipboard is empty!","HexCopy Error");
            }
            else
            {
               UltraEdit.messageBox("The user clipboard "+UltraEdit.clipboardIdx+" is empty!","HexCopy Error");
            }
         }
         else if (nOutputType == 1)
         {
            if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
            if (!UltraEdit.clipboardIdx)
            {
               UltraEdit.outputWindow.write("HexCopy: The Windows clipboard is empty!");
            }
            else
            {
               UltraEdit.outputWindow.write("HexCopy: The user clipboard "+UltraEdit.clipboardIdx+" is empty!");
            }
         }
         return ErrorCode;             // Nothing to do for this function.
      }
      nInput = 5;    // No further actions required to get source string.
   }
   else if (nInput < 4)                // Get string from active file.
   {
      // Nothing to do if no file is open.
      if (UltraEdit.document.length <= 0)
      {
         if (nOutputType == 2)
         {
            UltraEdit.messageBox("There is no file open according to UltraEdit.document.length!","HexCopy Error");
         }
         else if (nOutputType == 1)
         {
            if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
            UltraEdit.outputWindow.write("HexCopy: There is no file open according to UltraEdit.document.length!");
         }
         return ErrorCode;
      }

      // Should the clipboard be used temporarily?
      if (nInput > 1)
      {
         // Is this object supported by this version of the application?
         if (typeof(UltraEdit.clipboardContent) == "undefined")
         {
            if (nOutputType == 2)
            {
               UltraEdit.messageBox("Value "+DataSource+" for parameter DataSource not possible because UltraEdit.clipboardContent not supported!","HexCopy Error");
            }
            else if (nOutputType == 1)
            {
               if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
               UltraEdit.outputWindow.write("HexCopy: Value "+DataSource+" for parameter DataSource not possible because UltraEdit.clipboardContent not supported!");
            }
            return ErrorCode;    // Clipboard can't be used temporarily.
         }
      }

      // Is nothing selected?
      if (!UltraEdit.activeDocument.isSel())
      {
         if ((nInput == 1) || (nInput == 3))
         {
            if (nOutputType == 2)
            {
               UltraEdit.messageBox("There is nothing selected in the active file!","HexCopy Error");
            }
            else if (nOutputType == 1)
            {
               if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
               UltraEdit.outputWindow.write("HexCopy: There is nothing selected in the active file!");
            }
            return ErrorCode;
         }

         // Select the entire file content if nothing is selected.
         UltraEdit.activeDocument.selectAll();

         // Nothing to do if the active document is empty.
         if (!UltraEdit.activeDocument.isSel())
         {
            if (nOutputType == 2)
            {
               UltraEdit.messageBox("The active file is empty!","HexCopy Error");
            }
            else if (nOutputType == 1)
            {
               if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
               UltraEdit.outputWindow.write("HexCopy: The active file is empty!");
            }
            return ErrorCode;
         }

         // Get the (binary) data from the file into an array.
         if (!nInput) sSourceData = UltraEdit.activeDocument.selection;
         else
         {
            // Save content of current clipboard.
            sOutputText = UltraEdit.clipboardContent;
            // Copy selection into active clipboard.
            UltraEdit.activeDocument.copy();
            // Copy content of active clipboard into array.
            sSourceData = UltraEdit.clipboardContent;
            // Restore content of current clipboard.
            if (!sOutputText.length) UltraEdit.clearClipboard();
            else
            {
               UltraEdit.clipboardContent = sOutputText; sOutputText = "";
            }
         }
      }
      else  // Something is selected in the active document.
      {
         // Get the current file position (= offset of the first selected
         // char or byte) if this version of the application supports it.
         if (typeof(UltraEdit.activeDocument.currentPos) != "undefined")
         {
            nOffset = UltraEdit.activeDocument.currentPos;
         }
         // else nWriteOffset = 2;

         // Get the selected string directly or via the clipboard.
         if (nInput <= 1) sSourceData = UltraEdit.activeDocument.selection;
         else
         {
            sOutputText = UltraEdit.clipboardContent;
            UltraEdit.activeDocument.copy();
            sSourceData = UltraEdit.clipboardContent;
            if (!sOutputText.length) UltraEdit.clearClipboard();
            else
            {
               UltraEdit.clipboardContent = sOutputText; sOutputText = "";
            }
         }
      }
   }
   // ===== HexValues =====

   // Number of hexadecimal values (normally bytes) per line.
   if (typeof(HexValues) == "number")
   {
      if ((HexValues > 0) && (HexValues <= 512)) nHexCount = HexValues;
      else
      {
         if (nOutputType == 2)
         {
            UltraEdit.messageBox("Invalid value "+HexValues+" for parameter HexValues ignored!","HexCopy Warning");
         }
         else if (nOutputType == 1)
         {
            if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
            UltraEdit.outputWindow.write("HexCopy: Invalid value "+HexValues+" for parameter HexValues ignored!");
         }
      }
   }

   // ===== HexFormat =====

   // Format for the hexadecimal values.
   if ((typeof(HexFormat) == "boolean") || (typeof(HexFormat) == "number"))
   {
      if (!HexFormat) sHexValLeft = "";  // No space left the values.
   }
   else if (typeof(HexFormat) == "string")
   {
      // Empty string means also no space left the values.
      // A single character string defines the separator character.
      if (HexFormat.length <= 1) sHexValLeft = HexFormat;
      else
      {
         // Contains the string "%X" or "%x"?
         nIndex = HexFormat.search(/%X/i);
         if (nIndex < 0) sHexValLeft = ""; // No, don't output spaces.
         else
         {
            if (!nIndex) sHexValLeft = ""; // Nothing left the values!
            else sHexValLeft = HexFormat.substring(0,nIndex);

            // Output the hexadecimal values in lowercase?
            if (HexFormat.charAt(++nIndex) == 'x') bValuesUpper = false;

            // Something to output after every hexadecimal value?
            if (++nIndex < HexFormat.length)
            {
               sHexValRight = HexFormat.substr(nIndex);
            }
         }
      }
   }

   // ===== OffsetInfo =====

   // Evaluating the parameter OffsetInfo is similar to parameter HexFormat.
   if ((typeof(OffsetInfo) == "boolean") || (typeof(OffsetInfo) == "number"))
   {
      if(!OffsetInfo) nWriteOffset = 0;  // No output of offset info?
   }
   else if (typeof(OffsetInfo) == "string")
   {
      // Empty string disables offset information output.
      if (!OffsetInfo.length) nWriteOffset = 0;
      else
      {
         // Contains the string "%X" or "%x"?
         nIndex = OffsetInfo.search(/%X/i);
         if (nIndex >= 0)
         {
            if (nIndex) sOffsetLeft = OffsetInfo.substring(0,nIndex);

            // Output the hexadecimal offsets in uppercase?
            if (OffsetInfo.charAt(++nIndex) == 'X') bOffsetUpper = true;

            // Something to output after every offset?
            if (++nIndex >= OffsetInfo.length) sOffsetRight = "";
            else sOffsetRight = OffsetInfo.substr(nIndex);

            // Determine the number of digits for all offsets by temporary
            // converting the last (highest) offset to hexadecimal string
            // and get the length of this string.
            nIndex = nOffset + sSourceData.length - 1;
            sHexValue = nIndex.toString(16);
            nHexDigits = sHexValue.length;
         }
      }
   }

   // Should the offset info be written and debug messages are enabled and
   // a selection already existed, but the function could not determine the
   // position in the file because the version of the application does not
   // support the required document property, print here now the warning.
   // But that is not really possible because those versions of UE / UES
   // also do not support the commands to output the debug messages.
   // if (nWriteOffset == 2)
   // {
   //    if (nOutputType == 2)
   //    {
   //       UltraEdit.messageBox("The file offset can be wrong because UltraEdit.activeDocument.currentPos not supported!","HexCopy Warning");
   //    }
   //    else if (nOutputType == 1)
   //    {
   //       if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
   //       UltraEdit.outputWindow.write("HexCopy: The file offset can be wrong because UltraEdit.activeDocument.currentPos not supported!");
   //    }
   // }

   // If the text after the offset information ends with a space character
   // and the first character of the text left the hexadecimal values is
   // also a space, remove the space at end of text right the offset value
   // to avoid 2 spaces between offset info text and first value.
   if (sOffsetRight.length && sHexValLeft.length)
   {
      if ((sOffsetRight.charAt(sOffsetRight.length-1) == ' ') &&
          (sHexValLeft.charAt(0) == ' '))
      {
         sOffsetRight = sOffsetRight.substr(0,sOffsetRight.length-1);
      }
   }

   // ===== AsText =====

   // Evaluating now the parameter AsText.
   if ((typeof(AsText) == "boolean") || (typeof(AsText) == "number"))
   {
      if (!AsText) bWriteText = false;
   }
   else if (typeof(AsText) == "string")
   {
      // Empty string disables text output.
      if (!AsText.length) bWriteText = false;
      else
      {
         // Contains the string "%s"?
         nIndex = AsText.indexOf("%s");
         if (nIndex >= 0)
         {
            if (!nIndex) sTextLeft = ""; // Nothing left the characters!
            else sTextLeft = AsText.substring(0,nIndex);
            nIndex += 2;
            // Something to output after the characters before line ending?
            if (nIndex < AsText.length) sTextRight = AsText.substr(nIndex);
         }
      }
   }

   // ===== LineTerm =====

   // Evaluating the type of line termination. No check is done if the
   // selected line termination matches to the type of line termination in
   // an existing file when output should be written to an existing file.
   if (typeof(LineTerm) == "number")
   {
      if (LineTerm == 0) sLineEnd = "\r\n";     // DOS (default).
      else if (LineTerm == 1) sLineEnd = "\n";  // UNIX
      else if (LineTerm == 2) sLineEnd = "\r";  // MAC
      else if (LineTerm == 3) sLineEnd = "?";   // auto-detect
      else
      {
         if (nOutputType == 2)
         {
            UltraEdit.messageBox("Invalid value "+LineTerm+" for parameter LineTerm, using DOS!","HexCopy Warning");
         }
         else if (nOutputType == 1)
         {
            if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
            UltraEdit.outputWindow.write("HexCopy: Invalid value "+LineTerm+" for parameter LineTerm, using DOS!");
         }
      }
   }
   else if (typeof(LineTerm) == "boolean")
   {
      if (LineTerm) sLineEnd = "\n";  // DOS is default, true is for UNIX.
   }
   else if (typeof(LineTerm) == "string")
   {
      if ((LineTerm == "\r\n") || (LineTerm == "\n") ||
          (LineTerm == "\r") || (LineTerm == "?")) sLineEnd = LineTerm;
      else
      {
         if (nOutputType == 2)
         {
            UltraEdit.messageBox("Invalid string for parameter LineTerm, using DOS!","HexCopy Warning");
         }
         else if (nOutputType == 1)
         {
            if (!UltraEdit.outputWindow.visible) UltraEdit.outputWindow.showWindow(true);
            UltraEdit.outputWindow.write("HexCopy: Invalid string for parameter LineTerm, using DOS!");
         }
      }
   }

   // Should the type of line termination be automatically
   // detected and data got from active document?
   if ((sLineEnd == "?") && (nInput < 4))
   {
      if (UltraEdit.activeDocument.isHexModeOn()) sLineEnd = "\r\n";
   }
   // Line termination still not defined because data from clipboard,
   // from a passed string, or from a selection in text edit mode?
   if (sLineEnd == "?")
   {
      // Searching first explicitly for DOS line ending in
      // case of data are binary data got from the clipboard.
      if (sSourceData.indexOf("\r\n") >= 0) sLineEnd = "\r\n";
      else if (sSourceData.indexOf('\n') >= 0) sLineEnd = "\n";
      else if (sSourceData.indexOf('\r') >= 0) sLineEnd = "\r";
      else sLineEnd = "\r\n";
   }

   // Writing to a file with MAC line terminations requires a workaround
   // because of a bug of UltraEdit. When writing a long string with only
   // \r as line ending character into a MAC file UltraEdit does not display
   // the lines correct. Instead it shows the content as long line with
   // every \r displayed as character. The workaround is to use DOS line
   // terminations while building the text and write the output with DOS
   // line terminations to the file. If the function created a new file,
   // it is in this case created as DOS file and after writing the output
   // the file is converted to MAC. This bug was detected and reported to
   // IDM with UE for Windows v15.20.0.1022 and is possibly fixed in later
   // versions. In this case bMacWrite and the code blocks using this
   // variable would not be needed anymore.
   if ((nDestination > -2) && (sLineEnd == "\r"))
   {
      sLineEnd = "\r\n";
      bMacWrite = true;
   }

   // ===== Terminate =====

   // The last parameter evaluated is Terminate because the type of line
   // termination must be already defined to define the string correct.
   if ((typeof(Terminate) == "boolean") || (typeof(Terminate) == "number"))
   {
      // By default last line has no line termination. So
      // interesting is only if last line should have one.
      if(Terminate) sLastTerm = sLineEnd;
   }
   else if (typeof(Terminate) == "string")
   {
      // An empty string is the default. Anything else is interpreted as
      // text (footer, end of array) which should be appended to the output.
      if (Terminate.length) sLastTerm = Terminate;
   }

   // ===== HexCopy =====

   if (typeof(UltraEdit.activeDocumentIdx) == "undefined")
   {
      // Determine if 16 bit values are required because of a Unicode character
      // in the data string. At the same time check if any byte with value 0x80
      // to 0x9F is in the data stream. If this is the case a special copy to
      // clipboard must be used for UE for Windows < v16.00 and UES < v10.00.
      // Search for "bSpecialChar" to read more about this problem below.
      for (nIndex = 0; nIndex < sSourceData.length; nIndex++)
      {
         nCharCode = sSourceData.charCodeAt(nIndex);
         if (nCharCode > 255)
         {
            nValDigits = 4;
            if (bSpecialChar || (nDestination >= -1)) break;
         }
         else if ((nCharCode >= 0x80) && (nCharCode <= 0x9F))
         {
            bSpecialChar = true;
            if (nValDigits > 2) break;
         }
      }
   }
   else
   {
      // UE for Windows >= v16.00 and UES >= v10.00 do not require
      // anymore the workaround for bytes with value 0x80 to 0x9F.
      for (nIndex = 0; nIndex < sSourceData.length; nIndex++)
      {
         nCharCode = sSourceData.charCodeAt(nIndex);
         if (nCharCode > 255)
         {
            nValDigits = 4;
            break;
         }
      }
   }

   // Start output stream with offset information.
   nValueCount = nHexCount;

   // Convert the (binary) data to ASCII.
   nIndex = 0;
   do
   {
      // After every nValueCount values create a new line.
      if (nValueCount == nHexCount)
      {
         nValueCount = 0;  // Reset the value counter.

         // If (byte) index is not zero, append the ASCII representation of
         // the bytes to the hexadecimal values of the bytes and after the
         // optional text right the characters also the line termination.
         // The conversion is done line by line to improve performance on
         // large data by avoiding lots of small string concatenations on
         // the large output string.
         if (nIndex)
         {
            sOutputLine += sCharacters + sTextRight + sLineEnd;
            sOutputText += sOutputLine;
         }

         // Output also the offset information?
         if (!nWriteOffset) sOutputLine = "";
         else
         {
            // A line starts with the string left the offset value.
            sOutputLine = sOffsetLeft;

            // Convert the offset to a hexadecimal string.
            sHexValue = nOffset.toString(16);
            nDigitsCount = sHexValue.length;

            // Append leading zeros to line string until number of
            // leading zeros plus length of hexadecimal offset string
            // is equal the defined number of digits.
            while (nHexDigits > nDigitsCount++) sOutputLine += '0';

            // Append offset value and the defined text right of it.
            if (!bOffsetUpper) sOutputLine += sHexValue;
            else sOutputLine += sHexValue.toUpperCase()
            sOutputLine += sOffsetRight;
         }

         // Initialize the string for the bytes in ASCII/ANSI.
         if (bWriteText) sCharacters = sTextLeft;
      }

      // Start appending a new value with the defined text left to the value.
      sOutputLine += sHexValLeft;

      // Get value of current byte in the source data stream and
      // convert the byte/character value to a hexadecimal string.
      nCharCode = sSourceData.charCodeAt(nIndex);
      sHexValue = nCharCode.toString(16);

      // Append the required number of leading zeros in case the string with
      // the hexadecimal value has not 2 (normal) or 4 (Unicode) digits.
      nDigitsCount = sHexValue.length;
      while (nValDigits > nDigitsCount++) sOutputLine += '0';

      // Append the value string converted to uppercase if required.
      if (!bValuesUpper) sOutputLine += sHexValue;
      else sOutputLine += sHexValue.toUpperCase();

      // Append the byte value as ASCII/ANSI character to the ASCII string
      // or a point if the byte value is lower than 32 (= space character).
      if (bWriteText)
      {
         if ((nCharCode < 32) || (nCharCode > 255)) sCharacters += "."
         else sCharacters += sSourceData.charAt(nIndex);
      }

      nValueCount++;
      nOffset++;
      nIndex++;

      // Should something be appended right every hexadecimal value and
      // this was not the last value converted to hexadecimal ASCII?
      if (sHexValRight.length && (nIndex < sSourceData.length))
      {
         sOutputLine += sHexValRight;
      }
   }
   while (nIndex < sSourceData.length);

   // Omit last character of the string on last data value.
   if (sHexValRight.length > 1)
   {
      sOutputLine += sHexValRight.substr(0,sHexValRight.length-1);
   }
   // Instead the omitted character append a space character for alignment
   // if the output text has more than 1 line and the ASCII representation
   // of the bytes are appended too, or last line has also a line ending.
   if (sOutputText.length && sHexValRight.length &&
       (bWriteText || sLastTerm.length)) sOutputLine += ' ';

   // Finally append spaces to align the ASCII representation in the last
   // line. But the alignment is needed only when the output string already
   // has at least one line. The alignment is also needed when the output
   // has more than one line and the last line should be terminated.
   if (sOutputText.length && (bWriteText || sLastTerm.length))
   {
      nDigitsCount = sHexValLeft.length + nValDigits + sHexValRight.length;
      sHexValue = "";
      while(nDigitsCount--) sHexValue += ' ';
      nIndex = nValueCount;
      while (++nIndex <= nHexCount) sOutputLine += sHexValue;
      // Append also spaces to the ASCII representation if the last
      // line should be terminated and there are more than one line.
      if (bWriteText && sLastTerm.length)
      {
         while (++nValueCount <= nHexCount) sCharacters += ' ';
      }
   }
   // Append now the ASCII representation of the last line.
   if (bWriteText) sOutputLine += sCharacters + sTextRight;

   // Append the line termination (can be empty) to the last line.
   sOutputLine += sLastTerm;

   // Append the last line to the other output lines.
   sOutputText += sOutputLine;

   // ===== Output =====

   // Returning the output as string is the most easiest type of output.
   if (nDestination == -2) return sOutputText;

   // Writing the output to an already opened file is also very simple.
   if (nDestination >= 0) UltraEdit.document[nDestination].write(sOutputText);

   // Writing the output to a new file.
   else if (nDestination == -1)
   {
      // Always convert the file to ASCII/ANSI with DOS line endings
      // before continuing to make the file format independent of the
      // configuration settings for new files.
      UltraEdit.newFile();
      UltraEdit.activeDocument.unicodeToASCII();
      UltraEdit.activeDocument.unixMacToDos();

      // Is the type of line termination UNIX, convert the new file to UNIX.
      if (sLineEnd == "\n") UltraEdit.activeDocument.dosToUnix();

      // Write the output to the new file.
      UltraEdit.activeDocument.write(sOutputText);

      // Convert to MAC if the type of line termination is MAC.
      if (bMacWrite) UltraEdit.activeDocument.dosToMac();

      // Additionally save the new file?
      if (sFileName.length) UltraEdit.saveAs(sFileName == "?" ? "" : sFileName);
   }
   else  // Copying or appending the output to the active clipboard.
   {
      // Does this version of UltraEdit / UEStudio not support a direct
      // copy of a string to the active clipboard or contains characters
      // with byte code 0x80 to 0x9F?
      if ((typeof(UltraEdit.clipboardContent) == "undefined") ||
          (bSpecialChar && bWriteText))
      {
         // Must write the output into a new file, select the data,
         // copy or append the selection to active clipboard, close the
         // new file without saving and set focus back to current file.
         // The characters with byte code 0x80 to 0x9F must be encoded
         // with Unicode in the clipboard to be correct inserted to any
         // file. But the output string contains these characters in
         // ANSI. That is the reason why this workaround is used for
         // characters with byte code 0x80 to 0x9F.

         // Get the index of the active file to set focus back to
         // this file after closing the temporarily used new file.
         for (nIndex = 0; nIndex < UltraEdit.document.length; nIndex++)
         {
            if (UltraEdit.activeDocument.path == UltraEdit.document[nIndex].path) break;
         }
         UltraEdit.newFile();
         UltraEdit.activeDocument.unicodeToASCII();
         UltraEdit.activeDocument.unixMacToDos();
         if (sLineEnd == "\n") UltraEdit.activeDocument.dosToUnix();
         else if (sLineEnd == "\r") UltraEdit.activeDocument.dosToMac();
         UltraEdit.activeDocument.write(sOutputText);
         UltraEdit.activeDocument.selectAll();
         if (nDestination < -3) UltraEdit.activeDocument.copy();
         else UltraEdit.activeDocument.copyAppend();
         UltraEdit.closeFile(UltraEdit.activeDocument.path,2);
         UltraEdit.document[nIndex].setActive();
      }
      else  // The output can be copied or appended directly to the clipboard.
      {
         if (nDestination == -3) UltraEdit.clipboardContent += sOutputText;
         else
         {
            UltraEdit.clearClipboard(); // Not really needed, but more secure.
            UltraEdit.clipboardContent = sOutputText;
         }
      }
   }
   return true;
}  // End of function HexCopy


// === HexCopy Demonstration =================================================

// Code to demonstrate the usage of the function HexCopy.

var g_nDebugMessage=0;  // No debug messages at the beginning.

// Create array with all ASCII and ANSI Latin-1 characters starting
// with the space character. Well, it depends on your default code
// page if the characters 0x80 to 0xFF are really Latin-1 characters.
asAnsiCharTable = new Array(" !\"#$%&'()*+,-./", // 0x20 to 0x2F
                            "0123456789:;<=>?",  // 0x30 to 0x3F
                            "@ABCDEFGHIJKLMNO",  // 0x40 to 0x4F
                            "PQRSTUVWXYZ[\\]^_", // 0x50 to 0x5F
                            "`abcdefghijklmno",  // 0x60 to 0x6F
                            "pqrstuvwxyz{|}~",  // 0x70 to 0x7F
                            "",  // 0x80 to 0x8F
                            "",  // 0x90 to 0x9F
                            "",  // 0xA0 to 0xAF
                            "",  // 0xB0 to 0xBF
                            "",  // 0xC0 to 0xCF
                            "",  // 0xD0 to 0xDF
                            "",  // 0xE0 to 0xEF
                            ""); // 0xF0 to 0xFF

UltraEdit.ueReOn();
UltraEdit.insertMode();
if (typeof(UltraEdit.columnModeOff) == "function") UltraEdit.columnModeOff();
else if (typeof(UltraEdit.activeDocument.columnModeOff) == "function") UltraEdit.activeDocument.columnModeOff();
// Use the user clipboard 8 to demonstrate the usage of the function.
UltraEdit.selectClipboard(8);

// Create first a new Unicode file with DOS line endings to which all the
// examples are written for demonstrating the usage of the function.
UltraEdit.newFile();
var nDocIndex = UltraEdit.document.length-1;
var ExampleDoc = UltraEdit.document[nDocIndex];
ExampleDoc.ASCIIToUnicode();
ExampleDoc.unixMacToDos();
ExampleDoc.write("HexCopy with its default values as numbers, strings and booleans:\r\n\r\n");
ExampleDoc.write("HexCopy(Output=-4, DataSource=0, Terminate=0, HexValues=16, HexFormat=1, OffsetInfo=1, AsText=1, LineTerm=0);\r\n");
ExampleDoc.write("HexCopy(Output=-4, DataSource=0, Terminate=\"\", HexValues=16, HexFormat=\" \", OffsetInfo=\"yes\", AsText=\"yes\", LineTerm=\"\\r\\n\")\r\n");
ExampleDoc.write("HexCopy(Output=false, DataSource=false, Terminate=false, HexValues=16, HexFormat=true, OffsetInfo=true, AsText=true, LineTerm=false);\r\n\r\n");
ExampleDoc.write("=============================================================================\r\n\r\n");

// Example 1:
ExampleDoc.write("Example 1: HexCopy used to get hexadecimal values of a string.\r\n\r\n");

ExampleDoc.write("Sometimes a string should be found in a binary file and it would be good\r\n");
ExampleDoc.write("to run the search with the hexadecimal values. HexCopy can be used to copy\r\n");
ExampleDoc.write("a string you enter with hexadecimal values in ASCII to the clipboard. Let\r\n");
ExampleDoc.write("us try that.\r\n\r\n");

ExampleDoc.write("HexCopy(-4,UltraEdit.getString(\"Enter whatever you want:\",1),0,512,0,0,0);\r\n\r\n");

var sUserEnteredString = UltraEdit.getString("Enter whatever you want:",1);

ExampleDoc.write("You entered the string:\r\n\r\n");
ExampleDoc.write(sUserEnteredString);
ExampleDoc.write("\r\n\r\ncopied to clipboard by HexCopy as:\r\n\r\n");

// Because of bug of some versions of UltraEdit with command setActive
// resulting in a crash of UltraEdit let HexCopy write the output
// directly to this file instead of copying the output to the clipboard.
// HexCopy(-4,sUserEnteredString,0,512,0,0,0);
HexCopy(nDocIndex,sUserEnteredString,0,512,0,0,0);

//ExampleDoc.paste();
ExampleDoc.write("\r\n\r\n=============================================================================\r\n\r\n");


// Example 2:
ExampleDoc.write("Example 2: HexCopy used to get hexadecimal values of a string for an assembler.\r\n\r\n");

ExampleDoc.write("Fine! But you are an assembler programmer and need this string for a byte\r\n");
ExampleDoc.write("array. No problem with HexCopy. Just adapt the parameter HexFormat a little\r\n");
ExampleDoc.write("bit and this time write the result directly to this file with a DOS line\r\n");
ExampleDoc.write("termination.\r\n\r\n");

ExampleDoc.write("HexCopy(nDocIndex,sUserEnteredString,true,512,\"0%Xh,\",0,0);\r\n\r\n");

HexCopy(nDocIndex,sUserEnteredString,true,512,"0%Xh,",0,0);

ExampleDoc.write("\r\nPlease note that HexCopy omitted automatically the last comma.\r\n\r\n");
ExampleDoc.write("=============================================================================\r\n\r\n");


// Example 3:
ExampleDoc.write("Example 3: HexCopy used on an entire binary file with default settings.\r\n\r\n");
// Let's start with creating a new binary file with 256 bytes with the values 0 to 255.
UltraEdit.newFile();
UltraEdit.activeDocument.unicodeToASCII();
UltraEdit.activeDocument.unixMacToDos();
UltraEdit.activeDocument.write("                                "+asAnsiCharTable.join(""));
UltraEdit.activeDocument.top();
UltraEdit.activeDocument.hexOn();
// It depends on the version of UE if the following characters are
// overwriting the 32 spaces at top of the file or are inserted.
UltraEdit.activeDocument.write("00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F");
UltraEdit.activeDocument.write("10 11 12 13 14 15 16 17 18 19 1A 1B 1C 1D 1E 1F");
UltraEdit.activeDocument.top();
UltraEdit.activeDocument.hexOff();
UltraEdit.activeDocument.hexOn();
UltraEdit.activeDocument.top();
// To get identical output for all versions of UltraEdit the 32 spaces are
// deleted when they still exist in the binary file using a replace.
if (typeof(UltraEdit.activeDocument.findReplace.mode) != "undefined")
{
   UltraEdit.activeDocument.findReplace.mode=0;
}
UltraEdit.activeDocument.findReplace.matchCase=false;
UltraEdit.activeDocument.findReplace.matchWord=false;
UltraEdit.activeDocument.findReplace.regExp=true;
UltraEdit.activeDocument.findReplace.searchAscii=true;
UltraEdit.activeDocument.findReplace.searchDown=true;
UltraEdit.activeDocument.findReplace.preserveCase=false;
UltraEdit.activeDocument.findReplace.replaceAll=false;
UltraEdit.activeDocument.findReplace.replaceInAllOpen=false;
UltraEdit.activeDocument.findReplace.replace(" +"," ");
UltraEdit.activeDocument.findReplace.regExp=false;

ExampleDoc.write("Nothing is selected in a binary file opened in hex edit mode\r\n");
ExampleDoc.write("and containing 256 bytes with the values 0 to 255 (decimal).\r\n\r\n");
ExampleDoc.write("HexCopy();\r\n\r\n");

ExampleDoc.write("The output of the function was copied to user clipboard 8 which\r\n");
ExampleDoc.write("was selected by this script. It is pasted here so you can see it.\r\n\r\n");

// Because of bug of some versions of UltraEdit with command setActive
// resulting in a crash of UltraEdit let HexCopy write the output
// directly to this file instead of copying the output to the clipboard.
// HexCopy();
HexCopy(nDocIndex);

// ExampleDoc.paste();

ExampleDoc.write("|<-\r\n\r\n|<- was appended so you can see where the output of the function ended.\r\n\r\n");

ExampleDoc.write("The output is like using Ctrl+A and command Hex Copy Selected View. Using\r\n\r\n");
ExampleDoc.write("HexCopy(UltraEdit.activeDocument.path.replace(/\\.[^.]+$/,\".txt\"));\r\n\r\n");
ExampleDoc.write("would copy the content of the active binary file to an ASCII DOS file\r\n");
ExampleDoc.write("like displayed in hex edit mode and saved automatically with the name\r\n");
ExampleDoc.write("of the binary file, but with the file extension TXT in the same directory\r\n");
ExampleDoc.write("as the binary file. In other words the function can be used also as binary\r\n");
ExampleDoc.write("to ASCII conversion tool.\r\n\r\n");
ExampleDoc.write("=============================================================================\r\n\r\n");


// Example 4:
ExampleDoc.write("Example 4: HexCopy used on selection of binary file.\r\n\r\n");

ExampleDoc.write("This time the function should run on a selection without automatically\r\n");
ExampleDoc.write("selecting the entire file as done before when nothing is selected. And\r\n");
ExampleDoc.write("the output should be written directly to this file instead of copying\r\n");
ExampleDoc.write("to the active clipboard. And last the output should be without the\r\n");
ExampleDoc.write("ASCII representation of the bytes.\r\n\r\n");

ExampleDoc.write("HexCopy(nDocIndex,1,false,16,1,true,false);\r\n\r\n");

UltraEdit.activeDocument.top();
while (!HexCopy(nDocIndex,1,false,16,1,true,false))
{
   ExampleDoc.write("Ups, the function returned false. Why? Ah, forgot to select something.\r\n");
   ExampleDoc.write("Okay, scripts selects now something and runs HexCopy once again.\r\n\r\n");
   UltraEdit.activeDocument.gotoPage(40);
   UltraEdit.activeDocument.startSelect();
   UltraEdit.activeDocument.key("DOWN ARROW");
   for (var nCount=14; nCount; nCount--) UltraEdit.activeDocument.key("RIGHT ARROW");
   UltraEdit.activeDocument.endSelect();
   // The code above does not work with UE for Windows v13.00. Also searching
   // for hex values to select some bytes in hex edit mode is not working with
   // UE for Windows v13.00. As workaround let HexCopy run on the string
   // which should be selected now in the binary file in hex edit mode.
   if (!UltraEdit.activeDocument.isSel())
   {
      HexCopy(nDocIndex,"()*+,-./0123456789:;<=>",false,16,1,true,false);
      ExampleDoc.gotoLine(97,7);
      ExampleDoc.deleteText();
      ExampleDoc.deleteText();
      ExampleDoc.write("28");
      ExampleDoc.gotoLine(98,7);
      ExampleDoc.deleteText();
      ExampleDoc.deleteText();
      ExampleDoc.write("38");
      ExampleDoc.bottom();
      break;
   }
}

ExampleDoc.write("|<-\r\n\r\n|<- was appended so you can see where the output of the function ended.\r\n\r\n");
ExampleDoc.write("=============================================================================\r\n\r\n");


// Example 5:
ExampleDoc.write("Example 5: HexCopy used as C/C++ array builder.\r\n\r\n");

ExampleDoc.write("The output of Example 4 is nice, but what when binary data are needed for an\r\n");
ExampleDoc.write("array in C/C++, C#, JScript, etc. Can this function be used for that too?\r\n");
ExampleDoc.write("Yes! Let us convert the entire binary file into an initialized C/C++ array\r\n");
ExampleDoc.write("and append the closing bracket with the function.\r\n\r\n");
ExampleDoc.write("UltraEdit.document[nDocIndex].write(\"unsigned char CharTable[] = {\\r\\n\");\r\n\r\n");
ExampleDoc.write("HexCopy(nDocIndex,false,\" };\\r\\n\",16,\"0x%x,\",\"   /* %Xh: */ \",\" /* %s */\",\"\\r\\n\");\r\n\r\n");

UltraEdit.activeDocument.selectAll();
ExampleDoc.write("unsigned char CharTable[] = {\r\n");
HexCopy(nDocIndex,false," };\r\n",16,"0x%x,","   /* %Xh: */ "," /* %s */","\r\n");

ExampleDoc.write("|<-\r\n\r\n|<- was appended so you can see where the output of the function ended.\r\n\r\n");
ExampleDoc.write("Isn't that great? Look on the last hexadecimal value. There is no comma.\r\n");
ExampleDoc.write("The offset information is used here as array offset in comments. And the\r\n");
ExampleDoc.write("ASCII representations of the bytes are also in comments. The offset info\r\n");
ExampleDoc.write("is with uppercase letters and the byte values with lowercase letters. So\r\n");
ExampleDoc.write("it is also possible to change the case of the hexadecimal letters.\r\n\r\n");
ExampleDoc.write("=============================================================================\r\n\r\n");


// Example 6:
ExampleDoc.write("Example 6: HexCopy used to get the values of characters in a Unicode file.\r\n\r\n");

ExampleDoc.write("A Unicode file is open and you need the 16-bit hexadecimal values of some\r\n");
ExampleDoc.write("Unicode characters as ASCII string. That can be done also with HexCopy,\r\n");
ExampleDoc.write("but only if your version of UE / UES supports the clipboardContent object\r\n");
ExampleDoc.write("needed for that task.\r\n\r\n");
ExampleDoc.write("var sUnicodeValues = HexCopy(-2,3,false,1,\"\",\"\",\"\",\"\\n\");\r\n\r\n");

UltraEdit.activeDocument.hexOff();
UltraEdit.activeDocument.selectAll();
UltraEdit.activeDocument.write(" ");
UltraEdit.activeDocument.ASCIIToUnicode();
UltraEdit.activeDocument.hexOn();
UltraEdit.activeDocument.write("26 21 11 22 1A 22 1E 22");
UltraEdit.activeDocument.hexOff();
UltraEdit.activeDocument.top();
UltraEdit.activeDocument.findReplace.replace(" ", "");
UltraEdit.activeDocument.selectAll();

g_nDebugMessage = 1;  // Enable now debug messages to output window.
var sUnicodeValues = HexCopy(-2,3,false,1,"","","","\n");


if (!sUnicodeValues.length)
{
   ExampleDoc.write("Sorry, but HexCopy returned that your version of UltraEdit / UEStudio does\r\n");
   ExampleDoc.write("not support UltraEdit.clipboardContent and therefore it is not possible to\r\n");
   ExampleDoc.write("show you the hexadecimal values of the Unicode characters ");
   UltraEdit.activeDocument.copy();
   ExampleDoc.paste();
}
else
{
   UltraEdit.activeDocument.findReplace.replaceAll=true;
   UltraEdit.activeDocument.findReplace.regExp=true;
   UltraEdit.activeDocument.findReplace.replace("^(?^)", "   The value of Unicode character ^1 is ^p");
   UltraEdit.activeDocument.selectAll();
   UltraEdit.activeDocument.copy();
   ExampleDoc.paste();
   ExampleDoc.key("UP ARROW");
   ExampleDoc.key("UP ARROW");
   ExampleDoc.key("UP ARROW");
   ExampleDoc.key("UP ARROW");
   ExampleDoc.key("END");
   ExampleDoc.write(sUnicodeValues.substr(0,4)+".");
   ExampleDoc.key("DOWN ARROW");
   ExampleDoc.write(sUnicodeValues.substr(5,4)+".");
   ExampleDoc.key("DOWN ARROW");
   ExampleDoc.write(sUnicodeValues.substr(10,4)+".");
   ExampleDoc.key("DOWN ARROW");
   ExampleDoc.write(sUnicodeValues.substr(15,4)+".");
   ExampleDoc.bottom();
   ExampleDoc.write("\r\nTo get this ouput, HexCopy returned its output as string to the script and\r\n");
   ExampleDoc.write("a line-feed was used to separate the values which is the reason why only\r\n");
   ExampleDoc.write("1 value per line was written to the returned string. Also no offset and\r\n");
   ExampleDoc.write("no characters are added. That make it easy to get the output you see above\r\n");
   ExampleDoc.write("because the returned string is simply \"2126\\n2211\\n221A\\n221E\" which is\r\n");
   ExampleDoc.write("easy to split. HexCopy outputs automatically 4 digits per value if the\r\n");
   ExampleDoc.write("source data stream got from the active clipboard contains characters\r\n");
   ExampleDoc.write("with a value greater than 255");
}
ExampleDoc.write(".\r\n\r\n=============================================================================\r\n\r\n");


// Example 7:
ExampleDoc.write("Example 7: HexCopy used to get the hex values of a string in various formats.\r\n\r\n");

ExampleDoc.write("As last demonstration example a string is stored into a variable and the\r\n");
ExampleDoc.write("script calls HexCopy several times with different parameters to show you\r\n");
ExampleDoc.write("the output formatting options you have.\r\n\r\n");
ExampleDoc.write("var sExample = \"HexCopy Demonstration\";\r\n\r\n");

var sExample = "HexCopy Demonstration";

ExampleDoc.write("HexCopy(nDocIndex,sExample);\r\n\r\n");

HexCopy(nDocIndex,sExample);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("Additional spaces after the hexadecimal values are inserted to align the\r\n");
ExampleDoc.write("the characters on right side, but because of no line termination for the\r\n");
ExampleDoc.write("last line the output ends as marked with |<- directly after the characters.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,0,16,\".\");\r\n\r\n");

HexCopy(nDocIndex,sExample,0,16,".");

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("Like the first output, but with a different character left the bytes. On\r\n");
ExampleDoc.write("this output you can see in comparison to the first output another special\r\n");
ExampleDoc.write("behavior. Is the last character of the offset information a space and the\r\n");
ExampleDoc.write("first character of value is also a space, the space at end of offset info\r\n");
ExampleDoc.write("is ignored. Because every hexadecimal value in this output starts with a\r\n");
ExampleDoc.write("point instead of a space, the space at end of offset information is kept\r\n");
ExampleDoc.write("and the lines are 1 character longer.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,1);\r\n\r\n");

HexCopy(nDocIndex,sExample,1);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("Same as first output with the difference that the last line is terminated.\r\n");
ExampleDoc.write("Now HexCopy appended also trailing spaces after the characters on the last\r\n");
ExampleDoc.write("line which you can see for example when positioning the cursor at end of\r\n");
ExampleDoc.write("the line.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,false,16,true,true,false);\r\n\r\n");

HexCopy(nDocIndex,sExample,false,16,true,true,false);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("Standard output without ASCII representation and without terminating the\r\n");
ExampleDoc.write("last line. As marked with |<- the last line ends immediately after the\r\n");
ExampleDoc.write("last hexadecimal value in this case.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,true,16,true,true,false);\r\n\r\n");

HexCopy(nDocIndex,sExample,true,16,true,true,false);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("Standard output without ASCII representation, but this time with terminating\r\n");
ExampleDoc.write("the last line. Now HexCopy appends also spaces to the hexadecimal values in\r\n");
ExampleDoc.write("the last line to force identical line lengths for all lines.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,true,16,false,true,false);\r\n\r\n");

HexCopy(nDocIndex,sExample,true,16,false,true,false);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("The output is more compact when there is no space left the hex values.\r\n");
ExampleDoc.write("Note: The space left first value is from the offset information.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,0,16,0,0,0);\r\n\r\n");

HexCopy(nDocIndex,sExample,0,16,0,0,0);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("Without offset information the output is even compacter.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,0,16,1,0,0);\r\n\r\n");

HexCopy(nDocIndex,sExample,0,16,1,0,0);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("But with spaces the values are easier to read.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,1,32,1,0,0);\r\n\r\n");

HexCopy(nDocIndex,sExample,1,32,1,0,0);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("Now there are 32 bytes per line. But there is only 1 line. HexCopy\r\n");
ExampleDoc.write("doesn't need to append additional spaces in this case even if the\r\n");
ExampleDoc.write("last line is terminated.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,1,32,1,0,true);\r\n\r\n");

HexCopy(nDocIndex,sExample,1,32,1,0,true);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("When the output is within a single line, HexCopy also doesn't need to\r\n");
ExampleDoc.write("append additional spaces to the hex values as well as the characters.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,\"\",32,\"\",\"\",\"\");\r\n\r\n");

HexCopy(nDocIndex,sExample,"",32,"","","");

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("The most compact output consists of only hexadecimal values.\r\n\r\n\r\n");


ExampleDoc.write("HexCopy(nDocIndex,sExample,\";\",512,\"&#x%X;\",0,0);\r\n\r\n");

HexCopy(nDocIndex,sExample,";",512,"&#x%X;",0,0);

ExampleDoc.write("|<-\r\n\r\n");
ExampleDoc.write("HexCopy is used here for HTML encoding the string to make it really\r\n");
ExampleDoc.write("difficult for humans to read the string in the source code of a HTML\r\n");
ExampleDoc.write("file. HexCopy omits always the last character after the last hex value\r\n");
ExampleDoc.write("for building arrays. But the semicolon is needed here. No problem, it\r\n");
ExampleDoc.write("is appended with the termination string.\r\n\r\n\r\n");



ExampleDoc.write("End of demonstration.\r\n");
ExampleDoc.top();

UltraEdit.clearClipboard();
UltraEdit.selectClipboard(0);

UltraEdit.closeFile(UltraEdit.activeDocument.path,2);
// UltraEdit.document[nDocIndex].setActive();
