/* Script Name:   FindStringsWithLineNumbersExtended.js
   Creation Date: 2012-10-07
   Last Modified: 2022-05-29
   Copyright:     Copyright (c) 2022 by Mofi
   Original:      https://www.ultraedit.com/resources/scripts/FindStringsToNewFile.zip
                  http://forums.ultraedit.com/viewtopic.php?f=52&t=12866

Please read first FindStringsReadMe.txt before using this script.

This script is an extended version of script FindStringsWithLineNumbers.js
which can be used also on really large files as it processes the search and
output of the found strings in several steps on blocks of 50 000 lines to
avoid script execution errors.

Open the output window by clicking on Output Window in menu Window or by
moving mouse pointer over the tab of the output window (docked auto-hidden
output window) if the script execution is canceled unexpected.

It is possible that the script execution is canceled unexpected or UE/UES
crashed during script execution because of too much data copied to RAM.
In this case reduce the value of constant variable c_nMaxLoopRuns below,
set caret in the huge file to top and run the script again several times
on request until end of file is reached.

In comparison to FindStringsToNewFileExtended.js this script works on single
lines instead of entire file content. Therefore multi-line strings cannot be
searched. This script is also slower than FindStringsToNewFileExtended.js.

The main reason for searching on lines rather than entire file content
is the output of the line number left to found string. So this script
does not only list the found strings, it also reports in which line
every string was found.

The pattern for line information output left to found string can be
predefined here in the script with the 4 string variables at top of
the script.

The first variable is  sLineInfoPattern  which is by default an empty
string. In this case the user of the script will be asked for the pattern
during script execution. If the value of this variable is predefined for
example with "L#: " or "Line #### - " or "Found on line ######: ", the
script user does not need to enter the pattern during script execution.

The character # is used as placeholder of the line number within the line
information string output left to every found string. The number of # in
the pattern define the minimal width of the line number. For example using
#### in the pattern results in output the line numbers always with at least
4 digits. Leading zeros or spaces are used in this case for line numbers
smaller than 1000.

Second variable  sLeftLineNumber  defines the string left to line number
in case the pattern is an empty string and the script user does not enter
a pattern to simply use the default format as defined in the script.

Second variable  sRightLineNumber  defines the string right to line number
and left to found string in case the pattern is an empty string and the
script user does not enter a pattern to simply use the default as defined
in the script, or script user does not enter the placeholder character #.

If the script user enters a pattern string without line number placeholder
character, this string is interpreted as string to output left to line number.

The last variable  sLeadingChars  defines minimum width for the line numbers.
By default this is an empty string so that number of # in predefined pattern
or entered pattern during script execution defines the width of the line
numbers. In this case leading zeros are used for every # in the pattern.

But if the script should output line numbers always with at least X digits
even if the line information pattern string is entered by the script user
and the user enters just a single # or no # at all, the value of this string
with 1 or more ' ' (space) or '0' (zero) defines the minimum width of the
line number.

This script is copyrighted by Mofi for free usage by UE/UES users. The
author cannot be made responsible for any damage caused by this script.
You use it at your own risk. */


// Regular expression search string to use. The user of the script is asked
// to enter the regular expression search string if this string is empty.
var sSearchString = "";

// Regular expression replace string to specify a special output for the
// found strings in case of using marking (tagging) groups (round brackets).
// The user of the script is asked to enter the regular expression replace
// string for a special output format if this string is empty.
var sReplaceString = "";

var sLineInfoPattern = "";    // Pattern for line information can be predefined.
                              // If pattern is empty, script user will be asked.
var sLeftLineNumber  = "L";   // Default text left to line number in output.
var sRightLineNumber = ": ";  // Default text right to line number in output.
var sLeadingChars    = "";    // Define string with ' ' or '0' to define
                              // minimum width of line number.

// Constant maximum number of loop runs per script execution.
var c_nMaxLoopRuns = 8;
// Constant maximum number of lines to process in one loop run.
var c_nMaxLinesPerBlock = 50000;


var g_asFoundStrings;         // Define a typeless global variable used
                              // later for the array with the found strings.
var g_sConvertToMac = "2";    // By default no conversion to MAC on finishing.
var g_sLineTerm = "\r\n";     // Define DOS line termination type as default.


// Set this variable to value true if using UE for Windows > v18.00
// or UES > v12.00 for a better prompt text on asking user for line
// information pattern.
var bSupportsMultiLineUserPrompts = false;


// === OutputFoundStrings ====================================================

// A function is used to output the found strings in several blocks.
function OutputFoundStrings (nFoundCount)
{
   if (!nFoundCount)               // First output to file?
   {
      UltraEdit.newFile();         // Open a new file.
      // Insert a new line and get the new line character(s).
      UltraEdit.activeDocument.insertLine();
      UltraEdit.activeDocument.selectToTop();
      // Get line termination type of output file.
      g_sLineTerm = UltraEdit.activeDocument.selection;
      UltraEdit.activeDocument.deleteText();

      if (g_sLineTerm == "\r")     // Remembering MAC file format.
      {                            // If the new file is a MAC file
         g_sConvertToMac = "1";    // with just carriage returns
         g_sLineTerm = "\r\n";     // convert it temporary to DOS.
         UltraEdit.activeDocument.unixMacToDos();
      }
   }

   // If version of UE/UES supports direct write to clipboard,
   // use user clipboard 9 to paste the found strings into the
   // new file as this is much faster than using write command.
   if (typeof(UltraEdit.clipboardContent) == "string")
   {
      var nActiveClipboard = UltraEdit.clipboardIdx;
      UltraEdit.selectClipboard(9);
      UltraEdit.clipboardContent = g_asFoundStrings.join(g_sLineTerm);
      UltraEdit.activeDocument.paste();
      UltraEdit.clearClipboard();
      UltraEdit.selectClipboard(nActiveClipboard);
   }
   else UltraEdit.activeDocument.write(g_asFoundStrings.join(g_sLineTerm));

   // Append one more line termination to terminate also last line.
   UltraEdit.activeDocument.insertLine();
}


// === OutputProcessInfo =====================================================

// This function outputs process information message in output window.
function OutputProcessInfo (nLineCount, nFoundCount)
{
   if (UltraEdit.outputWindow)
   {
      if (UltraEdit.outputWindow.visible == false)
      {
         UltraEdit.outputWindow.showWindow(true);
      }
      var sProcessInfo = nLineCount.toString() + " line";
      if (nLineCount != 1) sProcessInfo += "s";
      sProcessInfo += " processed and " + nFoundCount.toString() + " string";
      if (nFoundCount != 1) sProcessInfo += "s";
      sProcessInfo += " found.";
      UltraEdit.outputWindow.write(sProcessInfo);
   }
}


// === FindStringsWithLineNumbersExtended ====================================

if (!UltraEdit.outputWindow)
{
   UltraEdit.newFile();
   UltraEdit.activeDocument.write("The script FindStringsWithLineNumbersExtended cannot be used with UE v13.00 and UES v6.20.");
}
else if (UltraEdit.document.length > 0)  // Is any file currently opened?
{
   UltraEdit.insertMode();               // Define environment for the script.
   if (typeof(UltraEdit.columnModeOff) == "function") UltraEdit.columnModeOff();
   else if (typeof(UltraEdit.activeDocument.columnModeOff) == "function") UltraEdit.activeDocument.columnModeOff();
   UltraEdit.activeDocument.hexOff();

   var nCount = 0;            // Total number of found strings.
   var nLineNumber = 1;       // Initialize line number variable.
   var sWidthChar = "0";      // Default leading character for line numbers.
   var sInputLineType = "";   // Line termination type of input file must be find out.
   var bClipContentSupport = (typeof(UltraEdit.clipboardContent) == "string") ? true : false;

   // Does used UE/UES support direct access to clipboard content?
   if (bClipContentSupport)
   {
      // In user clipboard 8 the variables of a previous run of this
      // script on this file are stored in case of a huge file which
      // requires several script executions to completely process it.
      // Determine first usage of this script on the active file.
      var nCurrentClipboard = UltraEdit.clipboardIdx;
      UltraEdit.selectClipboard(8);
      // Does user clipboard 8 contain any data?
      if (UltraEdit.clipboardContent.length)
      {
         // Does the string in user clipboard 8 start with name of the
         // script and contain also the special separator string "VaR?" ?
         if ((UltraEdit.clipboardContent.indexOf("FindStringsToNewFileExtended") == 0) &&
             (UltraEdit.clipboardContent.indexOf("VaR?") > 0))
         {
            // This script was executed most likely already before.
            var asVariables = UltraEdit.clipboardContent.split("VaR?");
            // Is the active file still the same file as on previous run.
            if ((asVariables[1] == UltraEdit.activeDocument.path) && (asVariables.length == 13))
            {
               // Yes, then this script run is a continued process of a file.
               // Get all variables from a previous run of this script.
               nCount = parseInt(asVariables[2],10);
               nLineNumber = parseInt(asVariables[3],10);
               g_sConvertToMac = asVariables[4];
               g_sLineTerm = asVariables[5];
               sLineInfoPattern = asVariables[6];
               sLeftLineNumber = asVariables[7];
               sRightLineNumber = asVariables[8];
               sLeadingChars = asVariables[9];
               sInputLineType = asVariables[10];
               sReplaceString = asVariables[11];
               sSearchString = asVariables[12];
               if (sLeadingChars.length) sWidthChar = sLeadingChars[0];
            }
            UltraEdit.clearClipboard();
         }
      }
      UltraEdit.selectClipboard(nCurrentClipboard);
   }

   // Must the script user enter the search string (or is it predefined)?
   if (!sSearchString.length)
   {
      sSearchString = UltraEdit.getString("Enter regular expression string to find text of interest:",1);
   }
   if (sSearchString.length)
   {
      // Search entire file line by line with a not case sensitive regular
      // expression search. As the script user can enter also an invalid
      // regular expression, catch this error and inform the script user
      // about this mistake.
      var bValidExpression = true;
      try
      {
         var rRegSearch = new RegExp(sSearchString,"gi");
      }
      catch (Error)
      {
         UltraEdit.messageBox("The entered search expression is an invalid regular expression.\n\n"+Error);
         bValidExpression = false;
      }

      if (bValidExpression)
      {
         // Was the script user asked already for the other input data?
         if (g_sConvertToMac == "2")
         {
            g_sConvertToMac = "0";  // No, do that now.
            var sPrompt = "";
            // Does the search string contain opening and closing parentheses?
            if (sSearchString.search(/.*\([^?].*\).*/) >= 0)
            {
               // Ask user for regular expression string determining output of
               // found string except the replace string is predefined in script.
               if (!sReplaceString.length)
               {
                  if (bSupportsMultiLineUserPrompts)
                  {
                     sPrompt = "Enter regular expression string for output.\n\nUse $1 for first marking group, $2 for second, ...";
                  }
                  else
                  {
                     sPrompt = "Enter regular expression string for output:";
                  }
                  sReplaceString = UltraEdit.getString(sPrompt,1);
               }
            }
            else sReplaceString = "";  // No replace string if no marking group.

            // Is the line information pattern already predefined in script?
            if (!sLineInfoPattern.length)
            {
               // No. Let script user enter the pattern. For UltraEdit for
               // Windows > v18.00 and UEStudio > v12.00 a multi-line prompt
               // text can be used. For previous versions just a short single
               // line prompt text can be used.
               if (bSupportsMultiLineUserPrompts)
               {
                  sPrompt = "Enter the pattern for line information.\n\nUse # 1 or more times as placeholder for line number.";
               }
               else
               {
                  sPrompt = "Enter the pattern for line information with # as placeholder:";
               }
               sLineInfoPattern = UltraEdit.getString(sPrompt,1);
            }

            // Define the character which is used to get identical width for the line numbers.
            if (sLeadingChars.length) sWidthChar = sLeadingChars[0];

            // If line information pattern is still an empty string,
            // use the predefined pattern as defined by sLeftLineNumber,
            // sLeadingChars and sRightLineNumber at top of script code.
            if (sLineInfoPattern.length)
            {
               // Find first placeholder character # in pattern.
               var nLineNumberPos = sLineInfoPattern.indexOf("#");
               // If not found at all, the pattern string is just the string left
               // to line number and sLeadingChars and sRightLineNumber define width
               // of line number and text between line number and found string.
               if (nLineNumberPos < 0) sLeftLineNumber = sLineInfoPattern;
               else
               {
                  // Everything left to first # is the string which is output
                  // left to line number. This can be also an empty string.
                  sLeftLineNumber = sLineInfoPattern.substr(0,nLineNumberPos);

                  var nLeadingChars = 0;
                  for (++nLineNumberPos; nLineNumberPos < sLineInfoPattern.length; nLineNumberPos++)
                  {
                     if (sLineInfoPattern[nLineNumberPos] != "#") break;
                     if (++nLeadingChars > sLeadingChars.length) sLeadingChars += sWidthChar;
                  }

                  // The remaining string after placeholder(s) # is
                  // output between line number and found string.
                  if (nLineNumberPos < sLineInfoPattern.length)
                  {
                     sRightLineNumber =  sLineInfoPattern.substr(nLineNumberPos);
                  }
               }
            }
            // Append one more character to string for leading chars of line number
            // to avoid subtraction or addition by 1 in the loop below to get correct
            // output of the line number string. This makes the loop a little bit faster.
            sLeadingChars += sWidthChar;
         }

         // Get document index of active file which is the input file.
         if (typeof(UltraEdit.activeDocumentIdx) == "number")
         {
            // There is a property available with UltraEdit
            // for Windows >= v16.00 or UEStudio >= v10.00.
            var nInputIndex = UltraEdit.activeDocumentIdx;
         }
         else
         {
            // Otherwise a loop must be used to determine document index of active file.
            for (var nInputIndex = 0; nInputIndex < UltraEdit.document.length; nInputIndex++)
            {
               if (UltraEdit.document[nInputIndex].path == UltraEdit.activeDocument.path) break;
            }
         }

         // Set caret either to top of the file or next line to process,
         // but only if UE for Windows >= v14.20 or UES >= v9.00 is used.
         if (bClipContentSupport)
         {
            if (nLineNumber < 2) UltraEdit.activeDocument.top();
            else
            {
               UltraEdit.activeDocument.gotoLine(nLineNumber,1);
               // Make the last file active as this should be the output
               // file if there was something found and written already.
               if (nCount > 0) UltraEdit.document[UltraEdit.document.length-1].setActive();
            }
         }
         else  // Otherwise continue script at current caret location.
         {     // The user must set caret to top of file before first run.
            nLineNumber = UltraEdit.activeDocument.currentLineNum;
         }

         // Is line termination type not determined already?
         if (!sInputLineType.length)
         {
            sInputLineType = "\r\n";
            // Determine line termination used currently in active file.
            if (typeof(UltraEdit.activeDocument.lineTerminator) == "number")
            {
               // The two lines below require UltraEdit for
               // Windows >= v16.00 or UEStudio >= v10.00.
               if (UltraEdit.activeDocument.lineTerminator == 1) sInputLineType = "\n";
               else if (UltraEdit.activeDocument.lineTerminator == 2) sInputLineType = "\r";
            }
            else  // This version of UE/UES does not offer line terminator property.
            {
               UltraEdit.activeDocument.selectLine();
               if (UltraEdit.activeDocument.isSel())
               {
                  if (UltraEdit.activeDocument.selection.indexOf(sInputLineType) < 0)
                  {
                     sInputLineType = "\n";          // Not DOS, perhaps UNIX.
                     if (UltraEdit.activeDocument.selection.indexOf(sInputLineType) < 0)
                     {
                        sInputLineType = "\r";       // Also not UNIX, perhaps MAC.
                        if (UltraEdit.activeDocument.selection.indexOf(sInputLineType) < 0)
                        {
                           sInputLineType = "\r\n";  // No line terminator, use DOS.
                        }
                     }
                  }
               }
               if (nLineNumber < 2) UltraEdit.activeDocument.top();
               else UltraEdit.activeDocument.gotoLine(nLineNumber,1);
            }
         }

         // Create a new, empty array for the strings found by the expression.
         g_asFoundStrings = new Array();
         var bEndOfFileReached = false;

         // Run the following loop X times or until entire file is processed.
         for (var nLoop = 0; nLoop < c_nMaxLoopRuns; nLoop++)
         {
            // Select the next block of lines in input file.
            UltraEdit.document[nInputIndex].gotoLineSelect(nLineNumber+c_nMaxLinesPerBlock,1);
            if (!UltraEdit.document[nInputIndex].isSel())
            {
               // Select everything to end of file.
               UltraEdit.document[nInputIndex].selectToBottom();
            }
            // If nothing could be selected, end of file is reached.
            if (!UltraEdit.document[nInputIndex].isSel())
            {
               bEndOfFileReached = true;
               break;
            }

            // Get all lines of selected block into memory as an array of strings.
            var asLines = UltraEdit.document[nInputIndex].selection.split(sInputLineType);

            // Cancel the selection using a moving command for downwards compatibility.
            // Usually the selection ends at beginning of a line and with setting the
            // caret to start of current line the selection can be cancelled. But if
            // the file has no line termination at end, it is necessary to move caret
            // to end of file to cancel the selection.
            if (UltraEdit.document[nInputIndex].cancelSelect) UltraEdit.document[nInputIndex].cancelSelect();
            else if (asLines.length <= c_nMaxLinesPerBlock) UltraEdit.document[nInputIndex].bottom();
            else UltraEdit.document[nInputIndex].gotoLine(nLineNumber+c_nMaxLinesPerBlock,1);

            // Remove last string from the array if it is an empty string
            // because of last block ended with a line termination.
            if (asLines[asLines.length-1] == "") asLines.pop();

            // Search now in the lines for the strings of interest using the expression.
            for (var nLineIndex = 0; nLineIndex < asLines.length; nLineIndex++)
            {
               // Search for strings in current line.
               var asStringsInLine = asLines[nLineIndex].match(rRegSearch);

               // Was any string found in this line?
               if (asStringsInLine != null)
               {
                  // Yes. Is a special output for the found strings requested?
                  if (sReplaceString.length)
                  {
                     for (var nIndex = 0; nIndex < asStringsInLine.length; nIndex++)
                     {
                        asStringsInLine[nIndex] = asStringsInLine[nIndex].replace(rRegSearch,sReplaceString);
                     }
                  }

                  // Convert line number as integer to line number string.
                  var sLineNumber = nLineNumber.toString();
                  // Insert leading chars according to defined minimum or pattern.
                  var nCharsCount = sLeadingChars.length - sLineNumber.length;
                  if (nCharsCount > 0) sLineNumber = sLeadingChars.substr(0,nCharsCount) + sLineNumber;
                  // Build the line information string for all found strings in this line.
                  var sLineInformation = sLeftLineNumber + sLineNumber + sRightLineNumber;

                  // Append all found strings with line information to array with found strings.
                  for (var nFoundIndex = 0; nFoundIndex < asStringsInLine.length; nFoundIndex++)
                  {
                      g_asFoundStrings.push(sLineInformation + asStringsInLine[nFoundIndex]);
                  }
               }
               nLineNumber++;
            }

            // Output all found strings with line information into new file.
            if (g_asFoundStrings.length)
            {
               OutputFoundStrings(nCount);
               nCount += g_asFoundStrings.length;
               // Free memory by deleting entire array content.
               g_asFoundStrings.splice(0,g_asFoundStrings.length);
            }
            OutputProcessInfo(nLineNumber-1,nCount);
         }

         // Is end of file not reached, another run of the script is necessary.
         if(!bEndOfFileReached)
         {
            // Does used UE/UES support direct access to clipboard content?
            if (bClipContentSupport)
            {
               UltraEdit.document[nInputIndex].setActive();
               nCurrentClipboard = UltraEdit.clipboardIdx;
               UltraEdit.selectClipboard(8);
               UltraEdit.clipboardContent = "FindStringsToNewFileExtendedVaR?";
               UltraEdit.clipboardContent += UltraEdit.activeDocument.path;
               UltraEdit.clipboardContent += "VaR?" + nCount.toString();
               UltraEdit.clipboardContent += "VaR?" + nLineNumber.toString();
               UltraEdit.clipboardContent += "VaR?" + g_sConvertToMac;
               UltraEdit.clipboardContent += "VaR?" + g_sLineTerm;
               UltraEdit.clipboardContent += "VaR?" + sLineInfoPattern;
               UltraEdit.clipboardContent += "VaR?" + sLeftLineNumber;
               UltraEdit.clipboardContent += "VaR?" + sRightLineNumber;
               UltraEdit.clipboardContent += "VaR?" + sLeadingChars;
               UltraEdit.clipboardContent += "VaR?" + sInputLineType;
               UltraEdit.clipboardContent += "VaR?" + sReplaceString;
               UltraEdit.clipboardContent += "VaR?" + sSearchString;
               UltraEdit.selectClipboard(nCurrentClipboard);
            }
            else if (nCount)
            {  // As with an older version of UE / UES the variables cannot
               // be remembered in user clipboard 8, the next run of the
               // script outputs again into a new file and therefore finish
               // the current output file.
               UltraEdit.activeDocument.top();
               if (g_sConvertToMac == "1") UltraEdit.activeDocument.dosToMac();
               UltraEdit.document[nInputIndex].setActive();
            }
            UltraEdit.messageBox("End of file not reached. Please run the script again after clicking on OK.");
         }
         else if (nCount)  // Was any string found?
         {
            // Set caret to top of file with found strings and convert
            // the file back to MAC if new file should be a MAC file.
            UltraEdit.activeDocument.top();
            if (g_sConvertToMac == "1") UltraEdit.activeDocument.dosToMac();
            // Inform user about number of strings found with the entered expression.
            UltraEdit.messageBox("Found "+nCount+" string"+(nCount==1?".":"s."));
         }
         else UltraEdit.messageBox("No string found!");
      }
   }
   else UltraEdit.messageBox("No regular expression string entered!");
}
else UltraEdit.messageBox("You should have a file opened when you run this script!");
