/* Script Name:   FindStringsWithLineNumbers.js
   Creation Date: 2012-10-07
   Last Modified: 2022-05-29
   Copyright:     Copyright (c) 2022 by Mofi
   Original:      https://www.ultraedit.com/resources/scripts/FindStringsToNewFile.zip
                  http://forums.ultraedit.com/viewtopic.php?f=52&t=12866

Please read first FindStringsReadMe.txt before using this script.

This script is like script FindStringsToNewFile.js with the difference that
line number information is also output on every line with a found string in
the new file.

The script can be used on small files as well as on larger files with
some MiB and is written for maximum performance by using only memory.

Open the output window by clicking on Output Window in menu Window or by
moving mouse pointer over the tab of the output window (docked auto-hidden
output window) if the script execution is canceled unexpected.

This script cannot be used for very large files. If you can read in output
window

      An error occurred on line 268:
        /
      Script failed.

an error occurred on the line with  UltraEdit.activeDocument.selection.split
because the file is too large respectively too many or too long strings are
found. Use the script FindStringsWithLineNumbersExtended.js which works also
for very large files, but is slower as it processes the large file in several
steps.

In comparison to FindStringsToNewFile.js this script works on single lines
instead of entire file content. Therefore multi-line strings cannot be
searched. And this script is also slower than FindStringsToNewFile.js.

The main reason for searching on lines rather than entire file content
is the output of the line number left to found string. So this script
does not only list the found strings, it also reports in which line
every string was found.

The pattern for line information output left to found string can be
predefined here in the script with the 4 string variables at top of
the script.

The first variable is  sLineInfoPattern  which is by default an empty
string. In this case the user of the script will be asked for the pattern
during script execution. If the value of this variable is predefined for
example with "L#: " or "Line #### - " or "Found on line ######: ", the
script user does not need to enter the pattern during script execution.

The character # is used as placeholder of the line number within the line
information string output left to every found string. The number of # in
the pattern define the minimal width of the line number. For example using
#### in the pattern results in output the line numbers always with at least
4 digits. Leading zeros or spaces are used in this case for line numbers
smaller than 1000.

Second variable  sLeftLineNumber  defines the string left to line number
in case the pattern is an empty string and the script user does not enter
a pattern to simply use the default format as defined in the script.

Second variable  sRightLineNumber  defines the string right to line number
and left to found string in case the pattern is an empty string and the
script user does not enter a pattern to simply use the default as defined
in the script, or script user does not enter the placeholder character #.

If the script user enters a pattern string without line number placeholder
character, this string is interpreted as string to output left to line number.

The last variable  sLeadingChars  defines minimum width for the line numbers.
By default this is an empty string so that number of # in predefined pattern
or entered pattern during script execution defines the width of the line
numbers. In this case leading zeros are used for every # in the pattern.

But if the script should output line numbers always with at least X digits
even if the line information pattern string is entered by the script user
and the user enters just a single # or no # at all, the value of this string
with 1 or more ' ' (space) or '0' (zero) defines the minimum width of the
line number.

This script is copyrighted by Mofi for free usage by UE/UES users. The
author cannot be made responsible for any damage caused by this script.
You use it at your own risk. */


// Regular expression search string to use. The user of the script is asked
// to enter the regular expression search string if this string is empty.
var sSearchString = "";

// Regular expression replace string to specify a special output for the
// found strings in case of using marking (tagging) groups (round brackets).
// The user of the script is asked to enter the regular expression replace
// string for a special output format if this string is empty.
var sReplaceString = "";

var sLineInfoPattern = "";    // Pattern for line information can be predefined.
                              // If pattern is empty, script user will be asked.
var sLeftLineNumber  = "L";   // Default text left to line number in output.
var sRightLineNumber = ": ";  // Default text right to line number in output.
var sLeadingChars    = "";    // Define string with ' ' or '0' to define
                              // minimum width of line number.

// Set this variable to value true if using UE for Windows > v18.00
// or UES > v12.00 for a better prompt text on asking user for line
// information pattern.
var bSupportsMultiLineUserPrompts = false;


// === OutputMessage =========================================================

// This function displays a message in a message box or write the message into
// a new file for users of UltraEdit for Windows v13.00 or UEStudio v6.20.
function OutputMessage (sMessageText)
{
   if (UltraEdit.messageBox) UltraEdit.messageBox(sMessageText);
   else
   {
      UltraEdit.newFile();
      UltraEdit.activeDocument.unixMacToDos();
      UltraEdit.activeDocument.unicodeToASCII();
      UltraEdit.activeDocument.write(sMessageText.replace(/\n/g,"\r\n"));
      UltraEdit.activeDocument.insertLine();
   }
}


// === FindStringsWithLineNumbers ============================================

if (UltraEdit.document.length > 0)     // Is any file currently opened?
{
   UltraEdit.insertMode();             // Define environment for the script.
   if (typeof(UltraEdit.columnModeOff) == "function") UltraEdit.columnModeOff();
   else if (typeof(UltraEdit.activeDocument.columnModeOff) == "function") UltraEdit.activeDocument.columnModeOff();
   UltraEdit.activeDocument.hexOff();

   // Must the script user enter the search string (or is it predefined)?
   if (!sSearchString.length)
   {
      sSearchString = UltraEdit.getString("Enter regular expression string to find text of interest:",1);
   }
   if (sSearchString.length)
   {
      // Search entire file line by line with a not case sensitive regular
      // expression search. As the script user can enter also an invalid
      // regular expression, catch this error and inform the script user
      // about this mistake.
      var bValidExpression = true;
      try
      {
         var rRegSearch = new RegExp(sSearchString,"gi");
      }
      catch (Error)
      {
         OutputMessage("The entered search expression is an invalid regular expression.\n\n"+Error);
         bValidExpression = false;
      }

      if (bValidExpression)
      {
         var sPrompt = "";

         // Does the search string contain opening and closing parentheses?
         if (sSearchString.search(/.*\([^?].*\).*/) >= 0)
         {
            // Ask user for regular expression string determining output of
            // found string except the replace string is predefined in script.
            if (!sReplaceString.length)
            {
               if (bSupportsMultiLineUserPrompts)
               {
                  sPrompt = "Enter regular expression string for output.\n\nUse $1 for first marking group, $2 for second, ...";
               }
               else
               {
                  sPrompt = "Enter regular expression string for output:";
               }
               // Ask user for regular expression string determining output of found string.
               sReplaceString = UltraEdit.getString(sPrompt,1);
            }
         }
         else sReplaceString = "";  // No replace string if no marking group.

         // Is the line information pattern already predefined in script?
         if (!sLineInfoPattern.length)
         {
            // No. Let script user enter the pattern. For UltraEdit for
            // Windows > v18.00 and UEStudio > v12.00 a multi-line prompt
            // text can be used. For previous versions just a short single
            // line prompt text can be used.
            if (bSupportsMultiLineUserPrompts)
            {
               sPrompt = "Enter the pattern for line information.\n\nUse # 1 or more times as placeholder for line number.";
            }
            else
            {
               sPrompt = "Enter the pattern for line information with # as placeholder:";
            }
            sLineInfoPattern = UltraEdit.getString(sPrompt,1);
         }

         // Define the character which is used to get identical width for the line numbers.
         var sWidthChar = sLeadingChars.length ? sLeadingChars[0] : "0";

         // If line information pattern is still an empty string,
         // use the predefined pattern as defined by sLeftLineNumber,
         // sLeadingChars and sRightLineNumber at top of script code.
         if (sLineInfoPattern.length)
         {
            // Find first placeholder character # in pattern.
            var nLineNumberPos = sLineInfoPattern.indexOf("#");
            // If not found at all, the pattern string is just the string left
            // to line number and sLeadingChars and sRightLineNumber define width
            // of line number and text between line number and found string.
            if (nLineNumberPos < 0) sLeftLineNumber = sLineInfoPattern;
            else
            {
               // Everything left to first # is the string which is output
               // left to line number. This can be also an empty string.
               sLeftLineNumber = sLineInfoPattern.substr(0,nLineNumberPos);

               var nLeadingChars = 0;
               for (++nLineNumberPos; nLineNumberPos < sLineInfoPattern.length; nLineNumberPos++)
               {
                  if (sLineInfoPattern[nLineNumberPos] != "#") break;
                  if (++nLeadingChars > sLeadingChars.length) sLeadingChars += sWidthChar;
               }

               // The remaining string after placeholder(s) # is
               // output between line number and found string.
               if (nLineNumberPos < sLineInfoPattern.length)
               {
                  sRightLineNumber =  sLineInfoPattern.substr(nLineNumberPos);
               }
            }
         }
         // Append one more character to string for leading chars of line number
         // to avoid subtraction or addition by 1 in the loop below to get correct
         // output of the line number string. This makes the loop a little bit faster.
         sLeadingChars += sWidthChar;

         // Determine line termination used currently in active file.
         var sLineTerm = "\r\n";
         if (typeof(UltraEdit.activeDocument.lineTerminator) == "number")
         {
            // The two lines below require UltraEdit for
            // Windows >= v16.00 or UEStudio >= v10.00.
            if (UltraEdit.activeDocument.lineTerminator == 1) sLineTerm = "\n";
            else if (UltraEdit.activeDocument.lineTerminator == 2) sLineTerm = "\r";
         }
         else  // This version of UE/UES does not offer line terminator property.
         {
            if (UltraEdit.activeDocument.selection.indexOf(sLineTerm) < 0)
            {
               sLineTerm = "\n";          // Not DOS, perhaps UNIX.
               if (UltraEdit.activeDocument.selection.indexOf(sLineTerm) < 0)
               {
                  sLineTerm = "\r";       // Also not UNIX, perhaps MAC.
                  if (UltraEdit.activeDocument.selection.indexOf(sLineTerm) < 0)
                  {
                     sLineTerm = "\r\n";  // No line terminator, use DOS.
                  }
               }
            }
         }

         // Get all lines of file into memory as an array of strings.
         UltraEdit.activeDocument.selectAll();
         // An error in the line below on script execution is caused usually by
         // a too large file for this script, use FindStringsToNewFileExtended.js.
         var asLines = UltraEdit.activeDocument.selection.split(sLineTerm);
         UltraEdit.activeDocument.top();  // Cancel selection.

         // Create a new, empty array for the strings found.
         var asFoundStrings = new Array();
         // Insert an empty string at beginning of lines array so that array index
         // starting with 0 is equal the line number starting with 1 for output.
         // The loop can then start with index 1 instead of 0 equal line number
         // which avoids incrementing nLineNumber by 1 for output string for
         // every line with at least 1 found string. This makes the loop faster.
         asLines.unshift("");

         // Search now in the lines for the strings of interest using the expression.
         for (var nLineNumber = 1; nLineNumber < asLines.length; nLineNumber++)
         {
            // Search for strings in current line.
            var asStringsInLine = asLines[nLineNumber].match(rRegSearch);

            // Was any string found in this line?
            if (asStringsInLine == null) continue;  // No.

            // Yes. Is a special output for the found strings requested?
            if (sReplaceString.length)
            {
               for (var nIndex = 0; nIndex < asStringsInLine.length; nIndex++)
               {
                  asStringsInLine[nIndex] = asStringsInLine[nIndex].replace(rRegSearch,sReplaceString);
               }
            }

            // Convert line number as integer to line number string.
            var sLineNumber = nLineNumber.toString();
            // Insert leading chars according to defined minimum or pattern.
            var nCharsCount = sLeadingChars.length - sLineNumber.length;
            if (nCharsCount > 0) sLineNumber = sLeadingChars.substr(0,nCharsCount) + sLineNumber;
            // Build the line information string for all found strings in this line.
            var sLineInformation = sLeftLineNumber + sLineNumber + sRightLineNumber;

            // Append all found strings with line information to array with found strings.
            for (var nFoundIndex = 0; nFoundIndex < asStringsInLine.length; nFoundIndex++)
            {
               asFoundStrings.push(sLineInformation + asStringsInLine[nFoundIndex]);
            }
         }

         if (asFoundStrings.length)    // Was any string found?
         {
            UltraEdit.newFile();       // Open a new file.
            // Insert a new line and get the new line character(s).
            UltraEdit.activeDocument.insertLine();
            UltraEdit.activeDocument.selectToTop();
            var sLineTerm = UltraEdit.activeDocument.selection;
            UltraEdit.activeDocument.deleteText();

            var bConvertToMac = false;
            if (sLineTerm == "\r")
            {                          // If the new file is a MAC file
               bConvertToMac = true;   // with just carriage returns
               sLineTerm = "\r\n";     // convert it temporary to DOS.
               UltraEdit.activeDocument.unixMacToDos();
            }

            // If version of UE/UES supports direct write to clipboard,
            // use user clipboard 9 to paste the found strings into the
            // new file as this is much faster than using write command.
            if (typeof(UltraEdit.clipboardContent) == "string")
            {
               var nActiveClipboard = UltraEdit.clipboardIdx;
               UltraEdit.selectClipboard(9);
               UltraEdit.clipboardContent = asFoundStrings.join(sLineTerm);
               UltraEdit.activeDocument.paste();
               UltraEdit.clearClipboard();
               UltraEdit.selectClipboard(nActiveClipboard);
            }
            else UltraEdit.activeDocument.write(asFoundStrings.join(sLineTerm));

            // Append one more line termination to terminate also last line.
            UltraEdit.activeDocument.insertLine();

            // Set caret to top of file with found strings and inform user
            // about number of strings found with the entered expression.
            UltraEdit.activeDocument.top();
            if (bConvertToMac) UltraEdit.activeDocument.dosToMac();
            var nCount = asFoundStrings.length;
            OutputMessage("Found "+nCount+" string"+(nCount==1?".":"s."));
         }
         else OutputMessage("No string found!");
      }
   }
   else OutputMessage("No regular expression string entered!");
}
else OutputMessage("You should have a file opened when you run this script!");
