/* Script Name:   FindStringsToNewFileExtended
   Creation Date: 2012-10-07
   Last Modified: 2022-05-29
   Copyright:     Copyright (c) 2022 by Mofi
   Original:      https://www.ultraedit.com/resources/scripts/FindStringsToNewFile.zip
                  http://forums.ultraedit.com/viewtopic.php?f=52&t=12866

Please read first FindStringsReadMe.txt before using this script.

This script is an extended version of script FindStringsToNewFile.js
which can be used also on really large files as it processes the search
and output of the found strings in several steps on blocks of 50 000 lines
to avoid script execution errors caused by running out of memory.

Open the output window by clicking on Output Window in menu Window or by
moving mouse pointer over the tab of the output window (docked auto-hidden
output window) if the script execution is canceled unexpected.

It is possible that the script execution is canceled unexpected or UE/UES
crashed during script execution because of too much data copied to RAM.
In this case reduce the value of constant variable c_nMaxLoopRuns below,
set caret in the huge file to top and run the script again several times
on request until end of file is reached.

This script is copyrighted by Mofi for free usage by UE/UES users. The
author cannot be made responsible for any damage caused by this script.
You use it at your own risk. */


// Regular expression search string to use. The user of the script is asked
// to enter the regular expression search string if this string is empty.
var sSearchString = "";

// Regular expression replace string to specify a special output for the
// found strings in case of using marking (tagging) groups (round brackets).
// The user of the script is asked to enter the regular expression replace
// string for a special output format if this string is empty.
var sReplaceString = "";


// Constant maximum number of loop runs per script execution.
var c_nMaxLoopRuns = 10;
// Constant maximum number of lines to process in one loop run.
var c_nMaxLinesPerBlock = 50000;


var g_asFoundStrings;      // Define a typeless global variable used
                           // later for the array with the found strings.
var g_sConvertToMac = "2"; // By default no conversion to MAC on finishing.
var g_sLineTerm = "\r\n";  // Define DOS line termination type as default.


// === OutputFoundStrings ====================================================

// A function is used to output the found strings in several blocks.
function OutputFoundStrings (nFoundCount)
{
   if (!nFoundCount)               // First output to file?
   {
      UltraEdit.newFile();         // Open a new file.
      // Insert a new line and get the new line character(s).
      UltraEdit.activeDocument.insertLine();
      UltraEdit.activeDocument.selectToTop();
      // Get line termination type of output file.
      g_sLineTerm = UltraEdit.activeDocument.selection;
      UltraEdit.activeDocument.deleteText();

      if (g_sLineTerm == "\r")     // Remembering MAC file format.
      {                            // If the new file is a MAC file
         g_sConvertToMac = "1";    // with just carriage returns
         g_sLineTerm = "\r\n";     // convert it temporary to DOS.
         UltraEdit.activeDocument.unixMacToDos();
      }
   }

   // If version of UE/UES supports direct write to clipboard,
   // use user clipboard 9 to paste the found strings into the
   // new file as this is much faster than using write command.
   if (typeof(UltraEdit.clipboardContent) == "string")
   {
      var nActiveClipboard = UltraEdit.clipboardIdx;
      UltraEdit.selectClipboard(9);
      UltraEdit.clipboardContent = g_asFoundStrings.join(g_sLineTerm);
      UltraEdit.activeDocument.paste();
      UltraEdit.clearClipboard();
      UltraEdit.selectClipboard(nActiveClipboard);
   }
   else UltraEdit.activeDocument.write(g_asFoundStrings.join(g_sLineTerm));

   // Append one more line termination to terminate also last line.
   UltraEdit.activeDocument.insertLine();
}


// === OutputProcessInfo =====================================================

// This function outputs process information message in output window.
function OutputProcessInfo (nLineCount, nFoundCount)
{
   if (UltraEdit.outputWindow)
   {
      if (UltraEdit.outputWindow.visible == false)
      {
         UltraEdit.outputWindow.showWindow(true);
      }
      var sProcessInfo = nLineCount.toString() + " line";
      if (nLineCount != 1) sProcessInfo += "s";
      sProcessInfo += " processed and " + nFoundCount.toString() + " string";
      if (nFoundCount != 1) sProcessInfo += "s";
      sProcessInfo += " found.";
      UltraEdit.outputWindow.write(sProcessInfo);
   }
}


// === FindStringsToNewFileExtended ==========================================

if (!UltraEdit.outputWindow)
{
   UltraEdit.newFile();
   UltraEdit.activeDocument.write("The script FindStringsToNewFileExtended cannot be used with UE v13.00 and UES v6.20.");
}
else if (UltraEdit.document.length > 0)  // Is any file currently opened?
{
   UltraEdit.insertMode();               // Define environment for the script.
   if (typeof(UltraEdit.columnModeOff) == "function") UltraEdit.columnModeOff();
   else if (typeof(UltraEdit.activeDocument.columnModeOff) == "function") UltraEdit.activeDocument.columnModeOff();
   UltraEdit.activeDocument.hexOff();

   var nCount = 0;         // Total number of found strings.
   var nLineNumber = 1;    // Initialize line number variable.
   var bClipContentSupport = (typeof(UltraEdit.clipboardContent) == "string") ? true : false;

   // Does used UE/UES support direct access to clipboard content?
   if (bClipContentSupport)
   {
      // In user clipboard 8 the variables of a previous run of this
      // script on this file are stored in case of a huge file which
      // requires several script executions to completely process it.
      // Determine first usage of this script on the active file.
      var nCurrentClipboard = UltraEdit.clipboardIdx;
      UltraEdit.selectClipboard(8);
      // Does user clipboard 8 contain any data?
      if (UltraEdit.clipboardContent.length)
      {
         // Does the string in user clipboard 8 start with name of the
         // script and contain also the special separator string "VaR?" ?
         if ((UltraEdit.clipboardContent.indexOf("FindStringsToNewFileExtended") == 0) &&
             (UltraEdit.clipboardContent.indexOf("VaR?") > 0))
         {
            // This script was executed most likely already before.
            var asVariables = UltraEdit.clipboardContent.split("VaR?");
            // Is the active file still the same file as on previous run.
            if ((asVariables[1] == UltraEdit.activeDocument.path) && (asVariables.length == 8))
            {
               // Yes, then this script run is a continued process of a file.
               // Get all variables from a previous run of this script.
               nCount = parseInt(asVariables[2],10);
               nLineNumber = parseInt(asVariables[3],10);
               g_sConvertToMac = asVariables[5];
               g_sLineTerm = asVariables[6];
               sReplaceString = asVariables[7];
               sSearchString = asVariables[8];
            }
            UltraEdit.clearClipboard();
         }
      }
      UltraEdit.selectClipboard(nCurrentClipboard);
   }

   // Must the script user enter the search string (or is it predefined)?
   if (!sSearchString.length)
   {
      sSearchString = UltraEdit.getString("Enter regular expression string to find text of interest:",1);
   }
   if (sSearchString.length)
   {
      // Search entire file with a not case sensitive regular expression.
      // As the script user can enter also an invalid regular expression,
      // catch this error and inform the script user about this mistake.
      var bValidExpression = true;
      try
      {
         var rRegSearch = new RegExp(sSearchString,"gi");
      }
      catch (Error)
      {
         UltraEdit.messageBox("The entered search expression is an invalid regular expression.\n\n"+Error);
         bValidExpression = false;
      }

      if (bValidExpression)
      {
         // Was the script user asked already for an expression for a special output?
         if (g_sConvertToMac == "2")
         {
            g_sConvertToMac = "0";
            // Does the search string contain opening and closing parentheses?
            if (sSearchString.search(/.*\([^?].*\).*/) >= 0)
            {
               // Ask user for regular expression string determining output of
               // found string except the replace string is predefined in script.
               if (!sReplaceString.length)
               {
                  sReplaceString = UltraEdit.getString("Enter regular expression string for output:",1);
               }
            }
            else sReplaceString = "";  // No replace string if no marking group.
         }

         // Get document index of active file which is the input file.
         if (typeof(UltraEdit.activeDocumentIdx) == "number")
         {
            // There is a property available with UltraEdit
            // for Windows >= v16.00 or UEStudio >= v10.00.
            var nInputIndex = UltraEdit.activeDocumentIdx;
         }
         else
         {
            // Otherwise a loop must be used to determine document index of active file.
            for (var nInputIndex = 0; nInputIndex < UltraEdit.document.length; nInputIndex++)
            {
               if (UltraEdit.document[nInputIndex].path == UltraEdit.activeDocument.path) break;
            }
         }

         // Set caret either to top of the file or next line to process,
         // but only if UE for Windows >= v14.20 or UES >= v9.00 is used.
         if (bClipContentSupport)
         {
            if (nLineNumber < 2) UltraEdit.activeDocument.top();
            else
            {
               UltraEdit.activeDocument.gotoLine(nLineNumber,1);
               // Make the last file active as this should be the output
               // file if there was something found and written already.
               if (nCount > 0) UltraEdit.document[UltraEdit.document.length-1].setActive();
            }
         }
         else  // Otherwise continue script at current caret location.
         {     // The user must set caret to top of file before first run.
            nLineNumber = UltraEdit.activeDocument.currentLineNum;
         }

         // Create a new, empty array for the strings found by the expression.
         g_asFoundStrings = new Array();
         var bEndOfFileReached = false;

         // Run the following loop X times or until entire file is processed.
         for (var nLoop = 0; nLoop < c_nMaxLoopRuns; nLoop++)
         {
            // Select the next block of lines in input file.
            nLineNumber += c_nMaxLinesPerBlock;
            UltraEdit.document[nInputIndex].gotoLineSelect(nLineNumber,1);
            if (!UltraEdit.document[nInputIndex].isSel())
            {
               // Select everything to end of file.
               UltraEdit.document[nInputIndex].selectToBottom();
            }
            // If nothing could be selected, end of file is reached.
            if (!UltraEdit.document[nInputIndex].isSel())
            {
               bEndOfFileReached = true;
               break;
            }

            g_asFoundStrings = UltraEdit.document[nInputIndex].selection.match(rRegSearch);

            // Output all found strings with line information into new file.
            if (g_asFoundStrings != null)
            {
               // Should a special output being used for the found strings?
               if (sReplaceString.length)
               {
                  for (var nIndex = 0; nIndex < g_asFoundStrings.length; nIndex++)
                  {
                     g_asFoundStrings[nIndex] = g_asFoundStrings[nIndex].replace(rRegSearch,sReplaceString);
                  }
               }
               OutputFoundStrings(nCount);
               nCount += g_asFoundStrings.length;
               // Free memory by deleting entire array content.
               g_asFoundStrings.splice(0,g_asFoundStrings.length);
            }

            // Cancel the selection using a moving command for downwards compatibility.
            // Usually the selection ends at beginning of a line and with setting the
            // caret to start of current line the selection can be cancelled. But if
            // the file has no line termination at end, it is necessary to move caret
            // to end of file to cancel the selection.
            if (UltraEdit.document[nInputIndex].cancelSelect) UltraEdit.document[nInputIndex].cancelSelect();
            else UltraEdit.document[nInputIndex].gotoLine(nLineNumber,1);
            if (UltraEdit.outputWindow && UltraEdit.document[nInputIndex].isEof())
            {
               nLineNumber = UltraEdit.document[nInputIndex].currentLineNum;
               if (!UltraEdit.document[nInputIndex].isColNum(1)) nLineNumber++;
            }
            OutputProcessInfo(nLineNumber-1,nCount);
         }

         // Is end of file not reached, another run of the script is necessary.
         if(!bEndOfFileReached)
         {
            // Does used UE/UES support direct access to clipboard content?
            if (bClipContentSupport)
            {
               UltraEdit.document[nInputIndex].setActive();
               nCurrentClipboard = UltraEdit.clipboardIdx;
               UltraEdit.selectClipboard(8);
               UltraEdit.clipboardContent = "FindStringsToNewFileExtendedVaR?";
               UltraEdit.clipboardContent += UltraEdit.activeDocument.path;
               UltraEdit.clipboardContent += "VaR?" + nCount.toString();
               UltraEdit.clipboardContent += "VaR?" + nLineNumber.toString();
               UltraEdit.clipboardContent += "VaR?" + g_sConvertToMac;
               UltraEdit.clipboardContent += "VaR?" + g_sLineTerm;
               UltraEdit.clipboardContent += "VaR?" + sReplaceString;
               UltraEdit.clipboardContent += "VaR?" + sSearchString;
               UltraEdit.selectClipboard(nCurrentClipboard);
            }
            else if (nCount)
            {  // As with an older version of UE / UES the variables cannot
               // be remembered in user clipboard 8, the next run of the
               // script outputs again into a new file and therefore finish
               // the current output file.
               UltraEdit.activeDocument.top();
               if (g_sConvertToMac == "1") UltraEdit.activeDocument.dosToMac();
               UltraEdit.document[nInputIndex].setActive();
            }
            UltraEdit.messageBox("End of file not reached. Please run the script again after clicking on OK.");
         }
         else if (nCount)  // Was any string found?
         {
            // Set caret to top of file with found strings and convert
            // the file back to MAC if new file should be a MAC file.
            UltraEdit.activeDocument.top();
            if (g_sConvertToMac == "1") UltraEdit.activeDocument.dosToMac();
            // Inform user about number of strings found with the entered expression.
            UltraEdit.messageBox("Found "+nCount+" string"+(nCount==1?".":"s."));
         }
         else UltraEdit.messageBox("No string found!");
      }
   }
   else UltraEdit.messageBox("No regular expression string entered!");
}
else UltraEdit.messageBox("You should have a file opened when you run this script!");
